<?php

declare(strict_types=1);

namespace IssetBV\Queuing\Message\Consumer;

use IssetBV\Queuing\Message\Consumer\Exception\ConsumerNotFoundException;
use IssetBV\Queuing\Message\MessageInterface;

class ConsumerContainer implements ConsumerContainerInterface
{
    /**
     * @var ConsumerInterface[]
     */
    private $consumers = [];

    /**
     * @param ConsumerInterface $consumer
     */
    public function add(ConsumerInterface $consumer)
    {
        $this->consumers[] = $consumer;
    }

    /**
     * @param MessageInterface $message
     *
     * @throws ConsumerNotFoundException
     *
     * @return ResponseInterface
     */
    public function consume(MessageInterface $message): ResponseInterface
    {
        $response = new ResponseAggregate();
        $consumersUsed = $this->fetchUsedConsumers($message);

        if (0 === \count($consumersUsed)) {
            throw new ConsumerNotFoundException('Consumer not found');
        }

        usort($consumersUsed, function (ConsumerInterface $a, ConsumerInterface $b) {
            $aWeight = $this->getWeight($a);
            $bWeight = $this->getWeight($b);
            if ($aWeight > $bWeight) {
                return -1;
            }

            if ($aWeight < $bWeight) {
                return 1;
            }

            return 0;
        });

        foreach ($consumersUsed as $consumer) {
            $response->addResponse($consumer->consume($message));
        }

        return $response;
    }

    private function getWeight(ConsumerInterface $consumer)
    {
        if ($consumer instanceof ConsumerWeightedInterface) {
            return $consumer->getWeight();
        }

        return 300;
    }

    private function fetchUsedConsumers(MessageInterface $message): array
    {
        $consumersUsed = [];
        foreach ($this->consumers as $consumer) {
            if ($consumer->consumes($message)) {
                $consumersUsed[] = $consumer;
            }
        }

        return $consumersUsed;
    }
}
