<?php

declare(strict_types=1);

namespace IssetBV\Queuing\Message\DeQueuer;

use IssetBV\Queuing\Message\Consumer\ConsumerContainerInterface;
use IssetBV\Queuing\Message\Consumer\Exception\ConsumerNotFoundException;
use IssetBV\Queuing\Message\Consumer\ResponseInterface;
use IssetBV\Queuing\Message\Decoder\DecoderContainerInterface;
use IssetBV\Queuing\Message\Decoder\DecoderData;
use IssetBV\Queuing\Message\Decoder\Exception\DecoderException;
use IssetBV\Queuing\Message\MessageInterface;
use IssetBV\Queuing\Queue\QueueInterface;
use IssetBV\Queuing\Queue\QueueMessageInterface;

class DeQueuer implements DeQueuerInterface
{
    /**
     * @var QueueInterface
     */
    private $queue;
    /**
     * @var DecoderContainerInterface
     */
    private $decoderContainer;
    /**
     * @var ConsumerContainerInterface
     */
    private $consumerContainer;

    public function __construct(
        DecoderContainerInterface $decoderContainer,
        ConsumerContainerInterface $consumerContainer,
        QueueInterface $queue
    ) {
        $this->queue = $queue;
        $this->decoderContainer = $decoderContainer;
        $this->consumerContainer = $consumerContainer;
    }

    public function dequeue(int $wait = 0): StatisticInterface
    {
        $dequeuerStats = new Statistic();
        $dequeuerStats->startTimer();
        $message = $this->queue->receive($wait);
        $dequeuerStats->stopTimer();
        if (null === $message) {
            return $dequeuerStats;
        }
        $dequeuerStats->addReceived();

        $this->handleMessage($message);

        return $dequeuerStats;
    }

    private function handleMessage(QueueMessageInterface $queueMessage)
    {
        try {
            $message = $this->decode($queueMessage);

            $response = $this->consumerContainer->consume($message);
            switch ($response->getState()) {
                case ResponseInterface::STATE_ACCEPT:
                    $this->queue->ack($queueMessage);
                    break;
                case ResponseInterface::STATE_REJECT:
                    $this->queue->reject($queueMessage, false);
                    break;
                case ResponseInterface::STATE_REQUEUE:
                    $this->queue->reject($queueMessage, true);
                    break;
                default:
                    throw new \LogicException('unknown response state: ' . $response->getState());
            }
        } catch (ConsumerNotFoundException $e) {
            $this->queue->reject($queueMessage, false);
            throw $e;
        } catch (DecoderException $e) {
            $this->queue->reject($queueMessage, false);
            throw $e;
        }
    }

    private function decode(QueueMessageInterface $queueMessage): MessageInterface
    {
        $type = $queueMessage->getHeader(MessageInterface::HEADER_TYPE);
        $version = $queueMessage->getHeader(MessageInterface::HEADER_VERSION);
        $body = $queueMessage->getBody();

        $data = new DecoderData($type, $version, $body);

        return $this->decoderContainer->decode($data);
    }
}
