<?php

declare(strict_types=1);

namespace IssetBV\Queuing\KafkaBundle\Kafka;

use IssetBV\Queuing\KafkaBundle\Kafka\Exception\KafkaException;
use IssetBV\Queuing\Queue\ConnectionInterface;
use IssetBV\Queuing\Queue\QueueInterface;
use IssetBV\Queuing\Queue\QueueMessageInterface;

class Queue implements QueueInterface
{
    /**
     * @var string
     */
    private $exchange;
    /**
     * @var string
     */
    private $queue;
    /**
     * @var ConnectionInterface
     */
    private $connection;
    /**
     * @var KafkaClient
     */
    private $kafkaClient;

    public function __construct(ConnectionInterface $connection, KafkaClient $kafkaClient, string $exchange, string $queue)
    {
        $this->exchange = $exchange;
        $this->queue = $queue;
        $this->connection = $connection;
        $this->kafkaClient = $kafkaClient;
    }

    /**
     * @param QueueMessageInterface $connectionMessage
     */
    public function ack(QueueMessageInterface $connectionMessage)
    {
    }

    /**
     * @param QueueMessageInterface $connectionMessage
     * @param bool $requeue
     *
     * @throws KafkaException
     */
    public function reject(QueueMessageInterface $connectionMessage, bool $requeue = false)
    {
        if (!$requeue) {
            return;
        }

        $offset = '' . $connectionMessage->getOption('offset');
        $this->resetTo($offset);
    }

    /**
     * @throws KafkaException
     */
    public function reset()
    {
        $response = $this->kafkaClient->reset($this->exchange, $this->queue);
        if (204 !== $response->getStatusCode()) {
            throw new KafkaException('Couldn\'t reset queue: ' . $response->getStatusCode() . ': ' . (string) $response->getBody());
        }
    }

    /**
     * @param string $offset
     *
     * @throws KafkaException
     */
    public function resetTo(string $offset)
    {
        $response = $this->kafkaClient->resetTo($this->exchange, $this->queue, $offset);
        if (204 !== $response->getStatusCode()) {
            throw new KafkaException('Couldn\'t reset queue to: ' . $offset . ': ' . $response->getStatusCode() . ': ' . (string) $response->getBody());
        }
    }

    /**
     * @param string $message
     * @param array $headers
     *
     * @throws KafkaException
     *
     * @return KafkaResponse
     */
    public function send(string $message, array $headers = []): KafkaResponse
    {
        $queueMessage = new QueueMessage($message, $headers);
        $response = $this->kafkaClient->produce($this->exchange, json_encode($queueMessage));
        if (201 !== $response->getStatusCode()) {
            throw new KafkaException('Couldn\'t produce message: ' . $response->getStatusCode() . ': ' . (string) $response->getBody());
        }

        return KafkaResponse::fromResponse(json_decode((string) $response->getBody(), true));
    }

    /**
     * @param int $wait
     *
     * @throws KafkaException
     *
     * @return QueueMessageInterface|null
     */
    public function receive(int $wait = 0)
    {
        $response = $this->kafkaClient->consume($this->exchange, $this->queue, $wait);

        switch ($response->getStatusCode()) {
            case 200:
                return Message::fromResponse(json_decode((string) $response->getBody(), true));
            case 404:
                return null;
            default:
                throw new KafkaException('couldn\'t receive message: ' . $response->getStatusCode() . ': ' . $response->getStatusCode());
        }
    }

    public function getConnection(): ConnectionInterface
    {
        return $this->connection;
    }

    public function warmUp(array $options = [])
    {
    }
}
