<?php

declare(strict_types=1);

namespace IssetBV\Queuing\KafkaBundle\Kafka;

use IssetBV\Http\Client\ClientInterface;
use IssetBV\Http\Client\Options;
use Psr\Http\Message\ResponseInterface;

class KafkaClient
{
    /**
     * @var string
     */
    private $connection;
    /**
     * @var string
     */
    private $baseUrl;
    /**
     * @var ClientInterface
     */
    private $client;
    /**
     * @var Options
     */
    private $options;
    private $headers = ['Content-Type' => 'application/json'];

    public function __construct(ClientInterface $client, string $baseUrl, string $connection)
    {
        $this->connection = $connection;
        $this->baseUrl = rtrim($baseUrl, '/');
        $this->client = $client;
    }

    public function reset(string $exchange, string $queue): ResponseInterface
    {
        $url = sprintf('%s/v1/kafka/%s/topic/%s/consume/%s/reset', $this->baseUrl, $this->connection, $exchange, $queue);
        $request = $this->client->create('PUT', $url, $this->headers);

        return $this->client->call($request, $this->getOptions());
    }

    public function resetTo(string $exchange, string $queue, string $offset): ResponseInterface
    {
        $url = sprintf('%s/v1/kafka/%s/topic/%s/consume/%s/reset/%s', $this->baseUrl, $this->connection, $exchange, $queue, $offset);
        $request = $this->client->create('PUT', $url, $this->headers);

        return $this->client->call($request, $this->getOptions());
    }

    public function produce(string $exchange, string $body): ResponseInterface
    {
        $url = sprintf('%s/v1/kafka/%s/topic/%s/produce', $this->baseUrl, $this->connection, $exchange);
        $request = $this->client->create('POST', $url, $this->headers, $body);

        return $this->client->call($request, $this->getOptions());
    }

    public function consume(string $exchange, string $queue, int $wait): ResponseInterface
    {
        $url = sprintf('%s/v1/kafka/%s/topic/%s/consume/%s?timeout=%s', $this->baseUrl, $this->connection, $exchange, $queue, $wait);
        $request = $this->client->create('GET', $url, $this->headers);

        return $this->client->call($request, $this->getOptions());
    }

    private function getOptions(): Options
    {
        if (null === $this->options) {
            $this->options = new Options();
            $this->options->setOption(Options::READ_TIMEOUT, 6);
            $this->options->setOption(Options::CONNECT_TIMEOUT, 6);
        }

        return $this->options;
    }
}
