<?php

declare(strict_types=1);

namespace IssetBV\Queuing\KafkaBundle\Kafka;

use Curl\Curl;
use IssetBV\Queuing\KafkaBundle\Kafka\Exception\KafkaException;
use IssetBV\Queuing\Queue\ConnectionInterface;
use IssetBV\Queuing\Queue\QueueInterface;
use IssetBV\Queuing\Queue\QueueMessageInterface;

class Queue implements QueueInterface
{
    /**
     * @var string
     */
    private $exchange;
    /**
     * @var string
     */
    private $queue;
    /**
     * @var string
     */
    private $connectionName;
    /**
     * @var string
     */
    private $baseUrl;
    /**
     * @var ConnectionInterface
     */
    private $connection;

    public function __construct(ConnectionInterface $connection, string $baseUrl, string $connectionName, string $exchange, string $queue)
    {
        $this->exchange = $exchange;
        $this->queue = $queue;
        $this->connectionName = $connectionName;
        $this->baseUrl = $baseUrl;
        $this->connection = $connection;
    }

    /**
     * @param QueueMessageInterface $connectionMessage
     */
    public function ack(QueueMessageInterface $connectionMessage)
    {
    }

    /**
     * @param QueueMessageInterface $connectionMessage
     * @param bool $requeue
     */
    public function reject(QueueMessageInterface $connectionMessage, bool $requeue = false)
    {
    }

    public function reset()
    {
        $url = sprintf('%s/kafka/%s/topic/%s/consume/%s/reset', 'v1', $this->connectionName, $this->exchange, $this->queue);
        $curl = $this->getCurl();
        $curl->put($url);

        if (0 !== $curl->errorCode) {
            throw new KafkaException('couldn\'t reset queue: ' . $curl->errorCode . ': ' . $curl->errorMessage);
        }
    }

    public function resetTo(string $offset)
    {
        $url = sprintf('%s/kafka/%s/topic/%s/consume/%s/reset/%s', 'v1', $this->connectionName, $this->exchange, $this->queue, $offset);
        $curl = $this->getCurl();
        $curl->put($url);

        if (0 !== $curl->errorCode) {
            throw new KafkaException('couldn\'t reset queue: ' . $curl->errorCode . ': ' . $curl->errorMessage);
        }
    }

    /**
     * @param string $message
     * @param array $headers
     *
     * @throws \IssetBV\Queuing\KafkaBundle\Kafka\Exception\KafkaException
     *
     * @return mixed
     */
    public function send(string $message, array $headers = [])
    {
        $queueMessage = new QueueMessage($message, $headers);
        $url = sprintf('%s/kafka/%s/topic/%s/produce', 'v1', $this->connectionName, $this->exchange);
        $curl = $this->getCurl();
        $curl->setUrl($url);
        $curl->setOpt(CURLOPT_POST, true);
        $curl->setOpt(CURLOPT_POSTFIELDS, json_encode($queueMessage));
        $curl->exec();

        if (0 !== $curl->errorCode) {
            throw new KafkaException('couldn\'t produce message: ' . $curl->errorCode . ': ' . $curl->errorMessage);
        }

        return $curl->response;
    }

    /**
     * @param int $wait
     *
     * @throws \IssetBV\Queuing\KafkaBundle\Kafka\Exception\KafkaException
     *
     * @return QueueMessageInterface|null
     */
    public function receive(int $wait = 0)
    {
        $url = sprintf('%s/kafka/%s/topic/%s/consume/%s?timeout=%s', 'v1', $this->connectionName, $this->exchange, $this->queue, $wait);
        $curl = $this->getCurl();
        $curl->get($url);
        if (404 === $curl->errorCode) {
            return null;
        }

        if (0 !== $curl->errorCode) {
            throw new KafkaException('couldn\'t receive message: ' . $curl->errorCode . ': ' . $curl->errorMessage);
        }

        return new Message($curl->response);
    }

    public function getConnection(): ConnectionInterface
    {
        return $this->connection;
    }

    private function getCurl(): Curl
    {
        $curl = new Curl($this->baseUrl);
        $curl->setHeader('Content-Type', 'application/json');
        $curl->setDefaultJsonDecoder(true);

        return $curl;
    }
}
