<?php

declare(strict_types=1);

namespace IssetBV\Queuing\AmqpBundle\AMQP;

use IssetBV\Queuing\Queue\QueueInterface;
use IssetBV\Queuing\Queue\QueueMessageInterface;
use PhpAmqpLib\Channel\AMQPChannel;
use PhpAmqpLib\Message\AMQPMessage as PhpAMQPMessage;

class Queue implements QueueInterface
{
    /**
     * @var string
     */
    private $exchange;
    /**
     * @var string
     */
    private $queue;
    /**
     * @var bool
     */
    private $exchangeDeclared = false;
    /**
     * @var bool
     */
    private $queueDeclared = false;
    /**
     * @var Connection
     */
    private $connection;
    /**
     * @var AMQPChannel
     */
    private $channel;
    /**
     * @var array
     */
    private $options;

    public function __construct(Connection $connection, string $exchange, string $queue, array $options = [])
    {
        $this->exchange = $exchange;
        $this->queue = $queue;
        $this->connection = $connection;
        $this->options = $options;
    }

    public function send(string $message, array $headers = [])
    {
        $this->declareExchange();

        $amqpMessage = new Message();
        $amqpMessage->setBody($message);

        foreach ($headers as $header => $headerValue) {
            $amqpMessage->addHeader(new Header($header, $headerValue));
        }

        $this->getChannel()->basic_publish($amqpMessage->getMessage(), $this->exchange);
    }

    /**
     * @param int $wait
     *
     * @return QueueMessageInterface|null
     */
    public function receive(int $wait = 0)
    {
        $this->queueDeclare();
        $message = $this->getChannel()->basic_get($this->queue);
        /* @var PhpAMQPMessage $message */
        if (null === $message) {
            return null;
        }

        return new Message($message);
    }

    public function ack(QueueMessageInterface $connectionMessage)
    {
        $this->getChannel()->basic_ack($connectionMessage->getOption('delivery_tag'));
    }

    public function reject(QueueMessageInterface $connectionMessage, bool $requeue = false)
    {
        $this->getChannel()->basic_reject($connectionMessage->getOption('delivery_tag'), $requeue);
    }

    public function declareExchange()
    {
        if ($this->exchangeDeclared) {
            return;
        }
        $this->getChannel()->exchange_declare($this->exchange, $this->options['exchange_type'] ?? 'fanout', false, false, false);
        $this->exchangeDeclared = true;
    }

    public function queueDeclare()
    {
        if ($this->queueDeclared) {
            return;
        }
        $this->declareExchange();
        $this->getChannel()->queue_declare($this->queue, false, false, false, false);
        $this->getChannel()->queue_bind($this->queue, $this->exchange);
        $this->queueDeclared = true;
    }

    /**
     * @return AMQPChannel
     */
    private function getChannel(): AMQPChannel
    {
        if (null === $this->channel) {
            $this->channel = $this->connection->getChannel();
        }

        return $this->channel;
    }
}
