<?php

namespace TPC\QueueingBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * QueueBase
 * @ORM\HasLifecycleCallbacks()
 * @Orm\MappedSuperclass(repositoryClass="TPC\QueueingBundle\Entity\QueueBaseRepository")
 */
class QueueBase
{

    /**
     * @var integer
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY") 
     */
    protected $id;

    /**
     * @ORM\ManyToOne(targetEntity="\TPC\QueueingBundle\Entity\QueueStatus", inversedBy="queueItems")
     * @ORM\JoinColumn(name="status_id", referencedColumnName="id")
     */
    protected $status;

    /**
     * @ORM\Column(name="message", type="string", length=255, nullable=true)
     */
    protected $message;

    /**
     * @ORM\Column(name="identifier", type="string", length=255, unique=true)
     */
    protected $identifier;

    /**
     * @var integer
      parent::__construct();
     * @Assert\Range(
     *      min = "1",
     *      max = "255",
     *      minMessage = "Minimal priority: 1",
     *      maxMessage = "Maximal priority 255",
     *      invalidMessage = "priority must be a number"
     * )
     * @ORM\Column(name="priority", type="integer")
     */
    protected $priority;

    /**
     * @var string
     * @Assert\Url(
     *      message = "Invalid callbackurl"
     * )
     * @ORM\Column(name="callbackurl", type="string", length=255, nullable=true)
     */
    protected $callbackurl;

    /**
     * @ORM\Column(name="logger_identifier", type="string", length=255, nullable=true)
     */
    protected $loggerIdentifier;

    /**
     * @ORM\ManyToOne(targetEntity="\TPC\QueueingBundle\Entity\QueueType", inversedBy="queueItems")
     * @ORM\JoinColumn(name="type_id", referencedColumnName="id")
     */
    protected $type;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="created", type="datetimetz", nullable=true)
     */
    private $created;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="updated", type="datetimetz", nullable=true)
     */
    private $updated;

    /**
     * @var integer
     *
     * @ORM\Column(name="tries", type="integer")
     */
    private $tries = 0;

    /**
     * @var integer
     *
     * @ORM\Column(name="max_tries", type="integer")
     */
    private $maxTries;

    public function setCreated($created)
    {
        $this->created = $created;

        return $this;
    }

    public function getCreated()
    {
        return $this->created;
    }

    public function setUpdated($updated)
    {
        $this->updated = $updated;

        return $this;
    }

    public function getUpdated()
    {
        return $this->updated;
    }

    public function setType(\TPC\QueueingBundle\Entity\QueueType $type = null)
    {
        $this->type = $type;

        return $this;
    }

    /**
     * Get status
     *
     * @return \TPC\QueueingBundle\Entity\QueueType
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * Get id
     *
     * @return integer 
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set identifier
     *
     * @param string identifier
     * @return QueueBase
     */
    public function setIdentifier($identifier)
    {
        $this->identifier = $identifier;

        return $this;
    }

    /**
     * Set status
     *
     * @param \TPC\QueueingBundle\Entity\QueueStatus $status
     * @return QueueBase
     */
    public function setStatus(\TPC\QueueingBundle\Entity\QueueStatus $status = null)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return \TPC\QueueingBundle\Entity\QueueStatus 
     */
    public function getStatus()
    {
        return $this->status;
    }

    public function setMessage($message)
    {
        $this->message = $message;

        return $this;
    }

    public function getMessage()
    {
        return $this->message;
    }

    /**
     * Get identifier
     *
     * @return string 
     */
    public function getIdentifier()
    {
        return $this->identifier;
    }

    /**
     * Set priority
     *
     * @param integer $priority
     * @return QueueBase
     */
    public function setPriority($priority)
    {
        $this->priority = $priority;

        return $this;
    }

    /**
     * Get priority
     *
     * @return integer 
     */
    public function getPriority()
    {
        return $this->priority;
    }

    /**
     * Set job
     *
     * @param string callbackurl
     * @return QueueBase
     */
    public function setCallbackurl($callbackurl)
    {
        $this->callbackurl = $callbackurl;

        return $this;
    }

    /**
     * Get callbackurl
     * @return string 
     */
    public function getCallbackurl()
    {
        return $this->callbackurl;
    }

    public function setLoggerIdentifier($loggerIdentifier)
    {
        $this->loggerIdentifier = $loggerIdentifier;

        return $this;
    }

    public function getLoggerIdentifier()
    {
        return $this->loggerIdentifier;
    }

    public function init($vars)
    {

        $this->setIdentifier(sha1(microtime(true) . mt_rand(10000, 90000)));
        $this->setPriority(1);

        foreach ($vars as $var => $data) {
            if (property_exists($this, $var)) {
                $setter = 'set' . ucfirst($var);
                if (method_exists($this, $setter)) {
                    $this->$setter($data);
                } else {
                    $this->$var = $data;
                }
            }
        }

        return true;
    }

    /**
     * Get all variables and values from entity
     * @return array
     */
    public function getItems()
    {

        $temp = get_object_vars($this);

        if (!is_null($temp)) {

            $result = array();
            foreach ($temp as $key => $value) {
                if (is_object($value) && method_exists($value, 'getName')) {
                    $result[$key] = $value->getName();
                } else {
                    $result[$key] = $value;
                }
            }
        }
        return $result;
    }

    /**
     * Update all values except status
     * @param String key
     * @param String value
     */
    public function update($key, $value)
    {

        if (property_exists($this, $key)) {
            $this->$key = $value;
            return true;
        } else {
            return false;
        }
    }

    /**
     * Set tries
     *
     * @param integer $tries
     * @return QueueBase
     */
    public function setTries($tries)
    {
        $this->tries = $tries;

        return $this;
    }

    /**
     * Get tries
     *
     * @return integer 
     */
    public function getTries()
    {
        return $this->tries;
    }

    /**
     * Set maxTries
     *
     * @param integer $maxTries
     * @return QueueBase
     */
    public function setMaxTries($maxTries)
    {
        $this->maxTries = $maxTries;

        return $this;
    }

    /**
     * Get maxTries
     *
     * @return integer 
     */
    public function getMaxTries()
    {
        return $this->maxTries;
    }

}
