<?php

namespace TPC\QueueingBundle\Handlers\ConnectorHandling;

use Doctrine\ORM\EntityManager;
use Symfony\Component\DependencyInjection\Container;

/**
 * Description of Connector
 *
 * @author Bart Malestein <bartmalestein88@gmail.com>
 */
class Connector
{

    private $em;
    private $ct;
    private $injection;
    private $observers = array();
    private $lastError;

    public function __construct(EntityManager $em, Container $ct, $injection = null)
    {
        $this->em        = $em;
        $this->ct        = $ct;
        $this->injection = $injection;
    }

    public function getLastError()
    {
        return $this->lastError;
    }

    public function setLastError($error)
    {
        $this->lastError = $error;
    }

    public function nextjob()
    {
        $repo = $this->ct->getParameter('Queue_repo');

        $isPaused = $this->em->getRepository('TPCQueueingBundle:QueueConfig')
                ->findOneByName('is_paused')
                ->getValue();

        if ($isPaused) {
            $this->setLastError('Queue is paused');
            return false;
        }

        $types = $this->em->getRepository('TPCQueueingBundle:QueueType')->findAll();

        foreach ($types as $type) {
            //Get the number of jobs in progress
            $busy = $this->em->getRepository($repo)
                    ->countBusy($repo, $type);

            //Exit if the limit is reached
            $limit = $type->getJobLimit();
            if ($busy >= $limit) {
                $this->setLastError('Job limit reached');
                continue;
            } else {

                
                $nexter = 'issetbv.queue.next.type.' . $type->getName();
                if ($this->ct->has($nexter)) {
                    $nextJobsHandler = $this->ct->get($nexter);
                } else {
                    $nextJobsHandler = $this->em->getRepository($repo);
                }

                foreach ($nextJobsHandler->getNext($type, $limit - $busy) as $nextJob) {

                    //If dependency is injected, perform code, else ignore and change job state
                    if ($this->injection) {
                        //this returns true when job is added succesfully
                        $next = $this->injection->nextJob($nextJob);

                        if ($next === True) {
                            //change job status to busy
                            $this->setJobBusy($nextJob);
                        } else if ($next === False) {
                            $this->setLastError('Could not add job');
                            $this->setJobFailed($nextJob);
                        }
                    } else {
                        $this->setJobBusy($nextJob);
                    }
                }
            }
        }
    }

    public function pauseJob($job)
    {
        if ($this->injection) {
            //this returns true when job is paused succesfully
            if ($this->injection->pauseJob($job)) {

//                //change job status to busy
//                $this->setJobPaused($job);
            } else {
                return false;
            }
        } else {
            $this->setJobPaused($job);
        }

        return true;
    }

    public function continueJob($job)
    {
        if ($this->injection) {
            //this returns true when job is resumed succesfully
            if ($this->injection->continueJob($job)) {

//                //change job status to busy
//                $this->setJobQueued($job);
            } else {
                return false;
            }
        } else {
            $this->setJobQueued($job);
        }

        return true;
    }

    public function deleteJob($job)
    {
        if ($this->injection) {

            if ($this->injection->deleteJob($job)) {
                
            } else {
                return false;
            }
        } else {
            
        }

        return true;
    }

    public function stopJob($job)
    {
        if ($this->injection) {

            if ($this->injection->stopJob($job)) {
                $this->setJobStopped($job);
            } else {
                return false;
            }
        } else {
            $this->setJobStopped($job);
        }

        return true;
    }

    public function forceStart($job)
    {
        if ($this->injection) {

            if ($this->injection->nextJob($job)) {
                $this->setJobBusy($job);
                return true;
            } else {
                $this->setJobFailed($job);
            }
        } else {
            $this->setJobFailed($job);
        }
        return false;
    }

    /**
     * Attach observer to notify callback on status change
     * @param Callback class $observer
     */
    public function attachObserver($observer)
    {
        $this->observers[] = $observer;
    }

    /**
     * Perform update on observer
     * @param job
     */
    private function notify($job)
    {
        foreach ($this->observers as $obs) {
            $obs->update($job);
        }
    }

    public function setJobBusy($job)
    {
        $this->setJobStatus($job, 'busy');
    }

    public function setJobPaused($job)
    {
        $this->setJobStatus($job, 'paused');
    }

    public function setJobRemoved($job)
    {
        $this->setJobStatus($job, 'removed');
    }

    public function setJobQueued($job)
    {
        $this->setJobStatus($job, 'queued');
    }

    public function setJobDone($job)
    {
        $this->setJobStatus($job, 'done');
    }

    public function setJobFailed($job)
    {
        $this->setJobStatus($job, 'failed');
    }

    public function setJobStopped($job)
    {
        $this->setJobStatus($job, 'stopped');
    }

    public function setJobStatus($job, $status)
    {
        $status = $this->em->getRepository('TPCQueueingBundle:QueueStatus')
                ->findOneByName($status);

        if ($status) {
            $job->setStatus($status);
            $this->em->flush();
            $this->notify($job);
        } else {
            return false;
        }
    }

    public function getBusyJobs()
    {
        $repo = $this->ct->getParameter('Queue_repo');

        //Get the number of jobs in progress
        $result = $this->em->getRepository($repo)
                ->getBusy($repo);
        if ($result) {
            return $result;
        } else {
            return false;
        }
    }

}

