<?php

declare(strict_types=1);

namespace IssetBV\PushNotificationBundle\Service\Windows;

use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Core\Message\Message;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelopeQueue;
use IssetBV\PushNotificationBundle\Service\Core\NotifierAbstract;
use IssetBV\PushNotificationBundle\Service\Windows\Message\WindowsMessage;
use IssetBV\PushNotificationBundle\Service\Windows\Message\WindowsMessageEnvelope;

/**
 * Class WindowsNotifier.
 */
class WindowsNotifier extends NotifierAbstract
{
    /**
     * @param Message $message
     *
     * @return bool
     */
    public function handles(Message $message): bool
    {
        return $message instanceof WindowsMessage;
    }

    /**
     * @param Message $message
     *
     * @return MessageEnvelope
     */
    protected function createMessageEnvelope(Message $message): MessageEnvelope
    {
        /* @var WindowsMessage $message */
        return new WindowsMessageEnvelope($message);
    }

    /**
     * @param Message $message
     * @param string $connectionName
     *
     * @throws \IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException
     * @throws \IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException
     *
     * @return MessageEnvelope
     */
    protected function sendMessage(Message $message, string $connectionName = null): MessageEnvelope
    {
        $envelope = $this->createMessageEnvelope($message);
        $connection = $this->getConnectionHandler()->getConnection($connectionName);

        $response = $connection->sendAndReceive($message);
        $envelope->setResponse($response);
        if ($response->isSuccess()) {
            $envelope->setState(MessageEnvelope::SUCCESS);
        } else {
            $envelope->setState(MessageEnvelope::FAILED);
        }

        return $envelope;
    }

    /**
     * @param string $connectionName
     * @param MessageEnvelopeQueue $queue
     *
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     */
    protected function flushQueueItem(string $connectionName, MessageEnvelopeQueue $queue)
    {
        if ($queue->isEmpty()) {
            return;
        }
        $connection = $this->getConnectionHandler()->getConnection($connectionName);
        /* @var WindowsConnection $connection */
        foreach ($queue->getQueue() as $messageEnvelope) {
            /* @var WindowsMessageEnvelope $messageEnvelope */
            $response = $connection->sendAndReceive($messageEnvelope->getMessage());
            $messageEnvelope->setResponse($response);
            if ($response->isSuccess()) {
                $messageEnvelope->setState(MessageEnvelope::SUCCESS);
            } else {
                $messageEnvelope->setState(MessageEnvelope::FAILED);
            }
        }
        $queue->reset();
    }
}
