<?php

declare(strict_types=1);

namespace IssetBV\PushNotificationBundle\Service\Android;

use IssetBV\PushNotificationBundle\Service\Android\Message\AndroidMessage;
use IssetBV\PushNotificationBundle\Service\Android\Message\AndroidMessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Core\Message\Message;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelopeQueue;
use IssetBV\PushNotificationBundle\Service\Core\NotifierAbstract;

/**
 * Class AndroidNotifier.
 */
class AndroidNotifier extends NotifierAbstract
{
    /**
     * @param Message $message
     *
     * @return bool
     */
    public function handles(Message $message): bool
    {
        return $message instanceof AndroidMessage;
    }

    /**
     * @param Message $message
     * @param string $connectionName
     *
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     *
     * @return MessageEnvelope
     */
    protected function sendMessage(Message $message, string $connectionName = null): MessageEnvelope
    {
        /* @var AndroidMessage $message */
        $connection = $this->getConnectionHandler()->getConnection($connectionName);
        $envelope = $this->createMessageEnvelope($message);
        $response = $connection->sendAndReceive($message);
        $envelope->setResponse($response);
        if ($response->isSuccess()) {
            $envelope->setState(MessageEnvelope::SUCCESS);
        } else {
            $envelope->setState(MessageEnvelope::FAILED);
        }

        return $envelope;
    }

    /**
     * @param Message $message
     *
     * @return MessageEnvelope
     */
    protected function createMessageEnvelope(Message $message): MessageEnvelope
    {
        /* @var AndroidMessage $message */
        return new AndroidMessageEnvelope($message);
    }

    /**
     * @param string $connectionName
     * @param MessageEnvelopeQueue $queue
     *
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     */
    protected function flushQueueItem(string $connectionName = null, MessageEnvelopeQueue $queue)
    {
        if ($queue->isEmpty()) {
            return;
        }
        $connection = $this->getConnectionHandler()->getConnection($connectionName);
        /* @var AndroidConnection $connection */
        foreach ($queue->getQueue() as $messageEnvelope) {
            /* @var AndroidMessageEnvelope $messageEnvelope */
            $response = $connection->sendAndReceive($messageEnvelope->getMessage());
            $messageEnvelope->setResponse($response);
            if ($response->isSuccess()) {
                $messageEnvelope->setState(MessageEnvelope::SUCCESS);
            } else {
                $messageEnvelope->setState(MessageEnvelope::FAILED);
            }
        }
        $queue->reset();
    }
}
