<?php

declare(strict_types=1);

namespace IssetBV\PushNotificationBundle\Service\Android;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use IssetBV\PushNotificationBundle\Service\Android\Message\AndroidMessage;
use IssetBV\PushNotificationBundle\Service\Core\Connection\Connection;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionResponseImpl;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Response;
use IssetBV\PushNotificationBundle\Service\LoggerTrait;

class AndroidConnection implements Connection
{
    use LoggerTrait;

    /**
     * @var Client
     */
    private $client;
    /**
     * @var string
     */
    private $type;
    /**
     * @var bool
     */
    private $default;
    /**
     * @var bool
     */
    private $dryRun;

    /**
     * AndroidConnection constructor.
     *
     * @param string $type
     * @param string $apiUrl
     * @param string $apiKey
     * @param int $timeout
     * @param bool $dryRun
     * @param bool $default
     */
    public function __construct(string $type, string $apiUrl, string $apiKey, int $timeout, bool $dryRun = false, bool $default = false)
    {
        $this->client = new Client([
            'base_uri' => rtrim($apiUrl, '/'),
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'key=' . $apiKey,
            ],
            'timeout' => $timeout,
        ]);
        $this->type = $type;
        $this->default = $default;
        $this->dryRun = $dryRun;
    }

    /**
     * @param MessageEnvelope $messageEnvelope
     *
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     *
     * @return Response
     */
    public function send(MessageEnvelope $messageEnvelope)
    {
        $message = $messageEnvelope->getMessage();
        /* @var AndroidMessage $message */
        $data = $this->getDataFromMessage($message);
        if ($this->dryRun) {
            $data['dry_run'] = true;
        }
        $requestData = json_encode($data);
        $request = new Request('POST', '', [], $requestData);
        $response = new ConnectionResponseImpl();
        $messageEnvelope->setRequest($requestData);
        $messageEnvelope->setResponse($response);
        try {
            $clientResponse = $this->client->send($request);
            $data = json_decode($clientResponse->getBody()->getContents(), true);
            $response->setResponse($data);
            $response->setSuccess($data['success'] > 0);
        } catch (RequestException $e) {
            $response->setErrorResponse($e->getMessage());
        }

        return $response;
    }

    /**
     * Send a message without waiting on response.
     *
     * @param MessageEnvelope $messageEnvelope
     *
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     */
    public function sendMessage(MessageEnvelope $messageEnvelope)
    {
        $this->send($messageEnvelope);
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return bool
     */
    public function isDefault(): bool
    {
        return $this->default;
    }

    private function getDataFromMessage(AndroidMessage $androidMessage): array
    {
        return $androidMessage->toArray();
    }
}
