<?php

declare(strict_types=1);

namespace IssetBV\PushNotificationBundle\Service\Apple;

use IssetBV\PushNotificationBundle\Service\Apple\Message\AppleMessage;
use IssetBV\PushNotificationBundle\Service\Core\Connection\Connection;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionExceptionImpl;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerExceptionImpl;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionResponseImpl;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Response;
use IssetBV\PushNotificationBundle\Service\LoggerTrait;

/**
 * Class Connection.
 */
class AppleConnection implements Connection
{
    use LoggerTrait;

    /**
     * Binary command to send a message to the APNS gateway (Internal use).
     */
    const BINARY_COMMAND = 1;

    /**
     * Binary size of a device token (Internal use).
     */
    const BINARY_DEVICE_TOKEN_SIZE = 32;

    /**
     * Command APNS sends back on error (Internal use).
     */
    const ERROR_RESPONSE_COMMAND = 8;

    /**
     * Size of the APNS error response (Internal use).
     */
    const ERROR_RESPONSE_SIZE = 6;
    /**
     * Minimum interval between sending two messages in microseconds (Internal use).
     */
    const SEND_INTERVAL = 10000;

    /**
     * Minimum interval to wait for a response (Internal use).
     */
    const READ_TIMEOUT = 1000000;

    /**
     * @var string
     */
    private $url;
    /**
     * @var string
     */
    private $type;
    /**
     * @var resource
     */
    private $connection = null;
    /**
     * @var string
     */
    private $pemKeyFile;
    /**
     * @var string
     */
    private $pemPasswordPhrase;
    /**
     * @var bool
     */
    private $default;

    /**
     * Connection constructor.
     *
     * @param string $url
     * @param string $type
     * @param string $pemKeyFile
     * @param string $pemPasswordPhrase
     * @param bool $default
     */
    public function __construct(string $url, string $type, string $pemKeyFile, string $pemPasswordPhrase = null, bool $default = false)
    {
        $this->url = $url;
        $this->type = $type;
        $this->pemKeyFile = $pemKeyFile;
        $this->pemPasswordPhrase = $pemPasswordPhrase;
        $this->default = $default;
    }

    public function __destruct()
    {
        $this->disconnect();
    }

    /**
     * @param MessageEnvelope $message
     */
    public function send(MessageEnvelope $message)
    {
        $this->sendMessage($message);
        $this->getResponseData();
    }

    /**
     * @throws ConnectionHandlerException
     *
     * @return Response
     */
    public function getResponseData(): Response
    {
        $connectionResponse = new ConnectionResponseImpl();
        $readStreams = [$this->getActiveConnection()];
        $write = $except = null;
        $streamsReadyToRead = stream_select($readStreams, $write, $except, 0, self::READ_TIMEOUT);
        if ($streamsReadyToRead > 0) {
            $errorResponse = stream_get_contents($this->getActiveConnection());
            $this->getLogger()->error('Send error: ' . $errorResponse);
            $this->disconnect();
            $response = @unpack('Ccommand/Cstatus/Nidentifier', $errorResponse);
            if (!empty($response)) {
                $connectionResponse->setErrorResponse($response);
            } else {
                $connectionResponse->setErrorResponse(null);
            }
        }

        return $connectionResponse;
    }

    /**
     * @param MessageEnvelope $messageEnvelope
     *
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     */
    public function sendMessage(MessageEnvelope $messageEnvelope)
    {
        $message = $messageEnvelope->getMessage();
        /* @var AppleMessage $message */
        $buildMessage = $this->buildMessage($message);
        $messageEnvelope->setRequest($buildMessage);
        $bytesSend = (int) fwrite($this->getActiveConnection(), $buildMessage, strlen($buildMessage));
        if (strlen($buildMessage) !== $bytesSend) {
            $this->disconnect();
            throw new ConnectionExceptionImpl();
        }
        usleep(self::SEND_INTERVAL);
    }

    /**
     * @throws ConnectionHandlerException
     */
    public function connect()
    {
        $this->disconnect();
        $streamContext = stream_context_create();
        stream_context_set_option($streamContext, 'ssl', 'local_cert', $this->pemKeyFile);

        if ($this->pemPasswordPhrase !== null) {
            stream_context_set_option($streamContext, 'ssl', 'passphrase', $this->pemPasswordPhrase);
        }
        $this->getLogger()->debug('opening connection to ' . $this->url . ' and pem key file: ' . $this->pemKeyFile . ' and using passphrase: ' . ($this->pemPasswordPhrase ? 'yes' : 'no'));
        $errorCode = null;
        $errorString = null;
        $connection = @stream_socket_client(
            $this->url,
            $errorCode,
            $errorString,
            (float) ini_get('default_socket_timeout'),
            STREAM_CLIENT_CONNECT,
            $streamContext
        );
        if ($connection === false) {
            if ($errorCode === 0) {
                $errorString = 'Error before connecting, please check your certificate and passphrase combo and the given CA certificate if any.';
            }
            $this->getLogger()->debug('Failed to connect to  with error #' . $errorCode . ' "' . $errorString . '".');
            throw new ConnectionHandlerExceptionImpl('Failed to connect to  with error #' . $errorCode . ' "' . $errorString . '".');
        }

        stream_set_blocking($connection, false);
        stream_set_write_buffer($connection, 0);
        $this->connection = $connection;
    }

    public function disconnect()
    {
        if (is_resource($this->connection)) {
            fclose($this->connection);
        }
        $this->connection = null;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return bool
     */
    public function isDefault(): bool
    {
        return $this->default;
    }

    /**
     * @throws ConnectionHandlerException
     *
     * @return resource
     */
    private function getActiveConnection()
    {
        if (!is_resource($this->connection)) {
            $this->connect();
        }

        return $this->connection;
    }

    /**
     * @param AppleMessage $message
     *
     * @return string
     */
    private function buildMessage(AppleMessage $message): string
    {
        $jsonMessage = json_encode($message->getMessage());
        $jsonMessageLength = strlen($jsonMessage);

        $payload =
            pack(
                'CNNnH*n',
                self::BINARY_COMMAND,
                $message->getIdentifier(),
                $message->getExpiresAt(),
                self::BINARY_DEVICE_TOKEN_SIZE,
                $message->getDeviceToken(),
                $jsonMessageLength
            );
        $payload .= $jsonMessage;

        return $payload;
    }
}
