<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Apple;

use Exception;
use IssetBV\PushNotificationBundle\Service\Apple\Message\AppleMessage;
use IssetBV\PushNotificationBundle\Service\Apple\Message\AppleMessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Core\Message\Message;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Message\MessageEnvelopeQueue;
use IssetBV\PushNotificationBundle\Service\Core\NotifierAbstract;

/**
 * Class AppleNotifier
 * @package IssetBV\PushNotificationBundle\Service\Apple
 */
class AppleNotifier extends NotifierAbstract
{

    /**
     * @param Message $message
     * @return bool
     */
    public function handles(Message $message): bool
    {
        return $message instanceof AppleMessage;
    }

    /**
     * @param Message $message
     * @param string|null $connectionName
     * @return MessageEnvelope
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     * @throws AppleNotifyFailedException
     */
    protected function sendMessage(Message $message, string $connectionName = null): MessageEnvelope
    {
        /* @var AppleMessage $message */
        try {
            $envelope = new AppleMessageEnvelope($message);
            $connection = $this->getConnectionHandler()->getConnection($connectionName);
            $connection->send($envelope);
            if ($envelope->getResponse()->isSuccess()) {
                $envelope->setState(MessageEnvelope::SUCCESS);
            } else {
                $envelope->setState(MessageEnvelope::FAILED);
            }
            return $envelope;
        } catch (Exception $e) {
            $this->getLogger()->error('Exception occurred sending an apple message: ' . $e->getMessage());
            throw new AppleNotifyFailedException($e->getMessage(), $e->getCode(), $e);
        }

    }

    /**
     * @param string $connectionName
     * @param MessageEnvelopeQueue $queue
     * @throws AppleNotifyFailedException
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     */
    protected function flushQueueItem(string $connectionName, MessageEnvelopeQueue $queue)
    {
        if ($queue->isEmpty()) {
            return;
        }
        $connection = $this->getConnectionHandler()->getConnection($connectionName);
        /* @var AppleConnection $connection */
        foreach ($queue->getQueue() as $item) {
            /* @var AppleMessageEnvelope $item */
            $connection->sendMessage($item);
        }
        $response = $connection->getResponseData();
        if ($response->isSuccess()) {
            $queue->setState(MessageEnvelope::SUCCESS);
            $queue->reset();
            return;
        }
        $error = $response->getResponse();
        if (!array_key_exists('identifier', $error)) {
            $queue->setState(MessageEnvelope::FAILED);
            $queue->reset();
            throw new AppleNotifyFailedException('Message gave an error but no response all messages marked as failed');
        }

        $identifier = $queue->removeToIdentifier($error['identifier']);
        $identifier->setResponse($response);
        $this->flushQueueItem($connectionName, $queue);

    }

    /**
     * @param Message $message
     * @return MessageEnvelope
     */
    protected function createMessageEnvelope(Message $message): MessageEnvelope
    {
        /* @var AppleMessage $message */
        return new AppleMessageEnvelope($message);
    }
}