<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Android;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use IssetBV\PushNotificationBundle\Service\Android\Message\AndroidMessage;
use IssetBV\PushNotificationBundle\Service\Core\Connection\Connection;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Core\Connection\ConnectionResponseImpl;
use IssetBV\PushNotificationBundle\Service\Core\Message\Message;
use IssetBV\PushNotificationBundle\Service\Core\Response;
use IssetBV\PushNotificationBundle\Service\LoggerTrait;

class AndroidConnection implements Connection
{

    use LoggerTrait;

    /**
     * @var Client
     */
    private $client;
    /**
     * @var string
     */
    private $type;
    /**
     * @var bool
     */
    private $default;

    /**
     * AndroidConnection constructor.
     * @param string $type
     * @param string $apiUrl
     * @param string $apiKey
     * @param int $timeout
     * @param bool $default
     */
    public function __construct(string $type, string $apiUrl, string $apiKey, int $timeout, bool $default = false)
    {
        $this->client = new Client([
            'base_uri' => rtrim($apiUrl, '/'),
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'key=' . $apiKey,
            ]
        ]);
        $this->type = $type;
        $this->default = $default;
    }

    /**
     * @param Message $message
     * @return Response
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     */
    public function send(Message $message): Response
    {
        /* @var AndroidMessage $message */
        $data = $message->toArray();
        if ($message->isDryRun()) {
            $data['dry_run'] = true;
        }
        $request = new Request('POST', '', [], json_encode($data));
        $response = new ConnectionResponseImpl();
        try {
            $clientResponse = $this->client->send($request);
            $data = json_decode($clientResponse->getBody()->getContents(), true);
            $response->setResponse($data);
            $response->setSuccess($data['success'] > 0);
        } catch (RequestException $e) {
            $response->setErrorResponse($e->getMessage());
        }
        return $response;
    }

    /**
     * Send a message without waiting on response
     *
     * @param Message $message
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     */
    public function sendMessage(Message $message)
    {
        /* @var AndroidMessage $message */
        $this->send($message);
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return bool
     */
    public function isDefault(): bool
    {
        return $this->default;
    }

}