<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Apple\Connection;

use IssetBV\PushNotificationBundle\Service\LoggerTrait;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerInterface;

/**
 * Class ConnectionHandler
 * @package IssetBV\PushNotificationBundle\Service\Apple\Connection
 */
class ConnectionHandler implements LoggerAwareInterface
{

    use LoggerTrait {
        setLogger as traitSetLogger;
    }

    /**
     * @var Connection[]
     */
    private $connections = [];

    /**
     * @var Connection
     */
    private $defaultConnection;

    /**
     * @param string $type
     * @return Connection
     * @throws ConnectionHandlerException
     */
    public function getConnection(string $type = null): Connection
    {
        if ($type === null) {
            return $this->getDefaultConnection();
        }
        if (!$this->hasConnectionType($type)) {
            $this->getLogger()->error('connection not found for type: ' . $type);
            throw new ConnectionHandlerException('connection not found for type: ' . $type);
        }
        return $this->connections[$type];
    }

    /**
     * @return Connection
     * @throws ConnectionHandlerException
     */
    public function getDefaultConnection(): Connection
    {
        if ($this->defaultConnection === null) {
            throw new ConnectionHandlerException('No default connection found');
        }
        return $this->defaultConnection;
    }

    /**
     * @param string $type
     * @throws ConnectionHandlerException
     */
    public function setDefaultConnectionByType(string $type)
    {
        $this->defaultConnection = $this->getConnection($type);
    }

    /**
     * Sets a logger instance on the object.
     *
     * @param LoggerInterface $logger
     *
     * @return void
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->traitSetLogger($logger);
        foreach ($this->connections as $connection) {
            $connection->setLogger($logger);
        }
    }

    /**
     * @param Connection $connection
     * @param bool $useLogger
     * @throws ConnectionHandlerException
     */
    public function addConnection(Connection $connection, bool $useLogger = true)
    {
        if ($useLogger) {
            $connection->setLogger($this->getLogger());
        }
        $this->connections[$connection->getType()] = $connection;
        if ($connection->isDefault()) {
            $this->setDefaultConnectionByType($connection->getType());
        }
    }

    /**
     * @param string $type
     * @return bool
     */
    public function hasConnectionType(string $type): bool
    {
        return array_key_exists($type, $this->connections);
    }
}