<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Apple\Connection;

use IssetBV\PushNotificationBundle\Service\LoggerTrait;
use Psr\Log\LoggerAwareInterface;

/**
 * Class Connection
 * @package IssetBV\PushNotificationBundle\Service\Apple\Connection
 */
class Connection implements LoggerAwareInterface
{

    use LoggerTrait;
    /**
     * Command APNS sends back on error (Internal use)
     */
    const ERROR_RESPONSE_COMMAND = 8;

    /**
     * Size of the APNS error response (Internal use)
     */
    const ERROR_RESPONSE_SIZE = 6;
    /**
     * Minimum interval between sending two messages in microseconds (Internal use)
     */
    const SEND_INTERVAL = 10000;

    /**
     * Minimum interval to wait for a response (Internal use)
     */
    const READ_TIMEOUT = 1000000;

    /**
     * @var string
     */
    private $url;
    /**
     * @var string
     */
    private $type;
    /**
     * @var resource
     */
    private $connection = null;
    /**
     * @var string
     */
    private $pemKeyFile;
    /**
     * @var string
     */
    private $pemPassphrase;
    /**
     * @var bool
     */
    private $default;

    /**
     * Connection constructor.
     * @param string $url
     * @param string $type
     * @param string $pemKeyFile
     * @param string $pemPassphrase
     * @param bool $default
     */
    public function __construct(string $url, string $type, string $pemKeyFile, string $pemPassphrase = null, bool $default = false)
    {
        $this->url = $url;
        $this->type = $type;
        $this->pemKeyFile = $pemKeyFile;
        $this->pemPassphrase = $pemPassphrase;
        $this->default = $default;
    }

    /**
     * @param string $message
     * @return ConnectionResponse
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     */
    public function send(string $message): ConnectionResponse
    {
        $this->sendMessage($message);
        return $this->getResponseData();
    }

    /**
     * @return ConnectionResponse
     * @throws ConnectionHandlerException
     */
    public function getResponseData(): ConnectionResponse
    {
        $connectionResponse = new ConnectionResponse();
        $readStreams = [$this->getActiveConnection()];
        $write = $except = null;
        $streamsReadyToRead = stream_select($readStreams, $write, $except, 0, self::READ_TIMEOUT);
        if ($streamsReadyToRead > 0) {
            $errorResponse = stream_get_contents($this->getActiveConnection());
            $this->getLogger()->error('Send error: ' . $errorResponse);
            $this->disconnect();
            $response = @unpack("Ccommand/Cstatus/Nidentifier", $errorResponse);
            if (!empty($response)) {
                $connectionResponse->setErrorResponse($response);
            } else {
                $connectionResponse->setErrorResponse(null);
            }
        }
        return $connectionResponse;
    }

    /**
     * @param string $message
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     */
    public function sendMessage(string $message)
    {
        $bytesSend = (int)fwrite($this->getActiveConnection(), $message, strlen($message));
        if (strlen($message) !== $bytesSend) {
            $this->disconnect();
            throw new ConnectionException();
        }
        usleep(self::SEND_INTERVAL);
    }

    /**
     * @return resource
     * @throws ConnectionHandlerException
     */
    private function getActiveConnection()
    {
        if (!is_resource($this->connection)) {
            $this->connect();
        }
        return $this->connection;
    }

    /**
     * @throws ConnectionHandlerException
     */
    public function connect()
    {
        $this->disconnect();
        $streamContext = stream_context_create();
        stream_context_set_option($streamContext, 'ssl', 'local_cert', $this->pemKeyFile);

        if ($this->pemPassphrase !== null) {
            stream_context_set_option($streamContext, 'ssl', 'passphrase', $this->pemPassphrase);
        }
        $this->getLogger()->debug('opening connection to ' . $this->url . ' and pem key file: ' . $this->pemKeyFile . ' and using passphrase: ' . ($this->pemPassphrase ? 'yes' : 'no'));
        $errorCode = null;
        $errorString = null;
        $connection = @stream_socket_client(
            $this->url,
            $errorCode,
            $errorString,
            (float)ini_get('default_socket_timeout'),
            STREAM_CLIENT_CONNECT,
            $streamContext
        );
        if ($connection === false) {
            if ($errorCode === 0) {
                $errorString = 'Error before connecting, please check your certificate and passphrase combo and the given CA certificate if any.';
            }
            $this->getLogger()->debug('Failed to connect to  with error #' . $errorCode . ' "' . $errorString . '".');
            throw new ConnectionHandlerException('Failed to connect to  with error #' . $errorCode . ' "' . $errorString . '".');
        }

        stream_set_blocking($connection, false);
        stream_set_write_buffer($connection, 0);
        $this->connection = $connection;

    }

    public function disconnect()
    {
        if (is_resource($this->connection)) {
            fclose($this->connection);
        }
        $this->connection = null;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return bool
     */
    public function isDefault(): bool
    {
        return $this->default;
    }

    public function __destruct()
    {
        $this->disconnect();
    }

}