<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Apple;

use Exception;
use IssetBV\PushNotificationBundle\Service\Apple\Connection\ConnectionException;
use IssetBV\PushNotificationBundle\Service\Apple\Connection\ConnectionHandler;
use IssetBV\PushNotificationBundle\Service\Apple\Connection\ConnectionHandlerException;
use IssetBV\PushNotificationBundle\Service\Apple\Message\AppleMessage;
use IssetBV\PushNotificationBundle\Service\Apple\Message\AppleMessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\Message;
use IssetBV\PushNotificationBundle\Service\Core\MessageEnvelope;
use IssetBV\PushNotificationBundle\Service\Core\MessageEnvelopeQueueImpl;
use IssetBV\PushNotificationBundle\Service\Core\NotifierAbstract;
use IssetBV\PushNotificationBundle\Service\Core\Response;
use Psr\Log\LoggerInterface;

/**
 * Class AppleNotifier
 * @package IssetBV\PushNotificationBundle\Service\Apple
 */
class AppleNotifier extends NotifierAbstract
{

    /**
     * Binary command to send a message to the APNS gateway (Internal use)
     */
    const BINARY_COMMAND = 1;

    /**
     * Binary size of a device token (Internal use)
     */
    const BINARY_DEVICE_TOKEN_SIZE = 32;

    /**
     * @var ConnectionHandler
     */
    private $connectionHandler;
    /**
     * @var MessageEnvelopeQueueImpl[]
     */
    protected $queues = [];

    /**
     * AppleNotifier constructor.
     * @param ConnectionHandler $connectionHandler
     */
    public function __construct(ConnectionHandler $connectionHandler)
    {
        $this->connectionHandler = $connectionHandler;
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        parent::setLogger($logger);
        $this->connectionHandler->setLogger($logger);
    }

    /**
     * @param Message $message
     * @return bool
     */
    public function handles(Message $message): bool
    {
        return $message instanceof AppleMessage;
    }

    /**
     * Flushes the queue to the notifier queues
     * @throws AppleNotifyFailedException
     * @throws ConnectionException
     * @throws ConnectionHandlerException
     */
    public function flushQueue()
    {

        if (empty($this->queues)) {
            return;
        }

        foreach ($this->queues as $connectionName => $queue) {
            $this->flushQueueItem($connectionName, $queue);
        }
    }

    /**
     * @param Message $message
     * @param string|null $connectionName
     * @return Response
     * @throws AppleNotifyFailedException
     */
    protected function sendMessage(Message $message, string $connectionName = null): Response
    {
        /* @var AppleMessage $message */
        try {
            $connection = $this->connectionHandler->getConnection($connectionName);
            $payload = $this->buildMessage($message);
            return $connection->send($payload);
        } catch (Exception $e) {
            $this->getLogger()->error('Exception occurred sending an apple message: ' . $e->getMessage());
            throw new AppleNotifyFailedException($e->getMessage(), $e->getCode(), $e);
        }

    }

    /**
     * @param AppleMessage $message
     * @return string
     */
    private function buildMessage(AppleMessage $message): string
    {
        $jsonMessage = json_encode($message->getMessage());
        $jsonMessageLength = strlen($jsonMessage);

        $payload =
            pack(
                'CNNnH*n',
                self::BINARY_COMMAND,
                $message->getIdentifier(),
                $message->getExpiresAt(),
                self::BINARY_DEVICE_TOKEN_SIZE,
                $message->getDeviceToken(),
                $jsonMessageLength
            );
        $payload .= $jsonMessage;
        return $payload;
    }

    /**
     * @param string $connectionName
     * @param MessageEnvelopeQueueImpl $queue
     * @throws AppleNotifyFailedException
     * @throws ConnectionHandlerException
     * @throws ConnectionException
     */
    private function flushQueueItem(string $connectionName, MessageEnvelopeQueueImpl $queue)
    {
        if ($queue->isEmpty()) {
            return;
        }
        $connection = $this->connectionHandler->getConnection($connectionName);
        foreach ($queue->getQueue() as $item) {
            $message = $item->getMessage();
            $payload = $this->buildMessage($message);
            $connection->sendMessage($payload);
        }
        $response = $connection->getResponseData();
        if ($response->isSuccess()) {
            $queue->setState(MessageEnvelope::SUCCESS);
            $queue->reset();
            return;
        }
        $error = $response->getErrorResponse();
        if (!array_key_exists('identifier', $error)) {
            $queue->setState(MessageEnvelope::FAILED);
            $queue->reset();
            throw new AppleNotifyFailedException('Message gave an error but no response all messages marked as failed');
        }

        $queue->removeToIdentifier($error['identifier']);
        $this->flushQueueItem($connectionName, $queue);

    }

    /**
     * @param Message $message
     * @param string|null $connectionName
     * @return MessageEnvelope
     * @throws ConnectionHandlerException
     */
    protected function addToQueue(Message $message, string $connectionName = null): MessageEnvelope
    {
        $envelope = new AppleMessageEnvelope($message);
        if ($connectionName === null) {
            $connectionName = $this->connectionHandler->getDefaultConnection()->getType();
        }
        if (!array_key_exists($connectionName, $this->queues)) {
            $this->queues[$connectionName] = new MessageEnvelopeQueueImpl();
        }
        $this->queues[$connectionName]->add($envelope);

        return $envelope;
    }

}