<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Apple\Message;

use DateTime;
use InvalidArgumentException;
use IssetBV\PushNotificationBundle\Service\Core\Message;

/**
 * Class AppleMessage
 * @package IssetBV\PushNotificationBundle\Service\Apple\Message
 */
abstract class AppleMessage implements Message
{

    /**
     * @var string
     */
    private $deviceToken;
    /**
     * @var int
     */
    private $expiresAt = 0;
    /**
     * @var string
     */
    private $connectionType;
    /**
     * @var int
     */
    private $identifier;

    public function __construct(string $connectionType, int $identifier)
    {
        $this->connectionType = $connectionType;
        $this->identifier = $identifier;
    }

    /**
     * @return int
     */
    public function getIdentifier(): int
    {
        return $this->identifier;
    }

    /**
     * @return string
     */
    public function getDeviceToken()
    {
        return $this->deviceToken;
    }

    /**
     * @param $deviceToken
     * @throws InvalidArgumentException
     */
    public function setDeviceToken(string $deviceToken)
    {
        // Check if a device token is given
        if ($deviceToken === null) {
            throw new InvalidArgumentException('No device token given.');
        }

        // Check if the device token is a valid hexadecimal string
        if (!ctype_xdigit($deviceToken)) {
            throw new InvalidArgumentException('Invalid device token given, no hexadecimal: ' . $deviceToken);
        }

        // Check if the length of the device token is correct
        if (strlen($deviceToken) !== 64) {
            throw new InvalidArgumentException('Invalid device token given, incorrect length: ' . $deviceToken . ' (' . strlen($deviceToken) . ')');
        }

        $this->deviceToken = $deviceToken;
    }

    /**
     * @return int
     */
    public function getExpiresAt(): int
    {
        return $this->expiresAt;
    }

    /**
     * @param DateTime $expiresAt
     */
    public function setExpiresAt(DateTime $expiresAt)
    {
        $this->expiresAt = $expiresAt->format('U');
    }

    /**
     * @return string
     */
    public function getConnectionType(): string
    {
        return $this->connectionType;
    }

    /**
     * @param string $connectionType
     */
    public function setConnectionType(string $connectionType)
    {
        $this->connectionType = $connectionType;
    }

    abstract public function getMessage(): array;

}