<?php
declare(strict_types = 1);

namespace IssetBV\PushNotificationBundle\Service\Apple;

use Exception;
use IssetBV\PushNotificationBundle\Service\Apple\Connection\ConnectionHandler;
use IssetBV\PushNotificationBundle\Service\Apple\Message\AppleMessage;
use IssetBV\PushNotificationBundle\Service\Core\Message;
use IssetBV\PushNotificationBundle\Service\Core\NotifierAbstract;
use IssetBV\PushNotificationBundle\Service\Core\NotifyFailedException;
use IssetBV\PushNotificationBundle\Service\Core\Response;
use Psr\Log\LoggerInterface;

/**
 * Class IOSNotifier
 * @package IssetBV\PushNotificationBundle\Service\IOS
 */
class AppleNotifier extends NotifierAbstract
{

    /**
     * Binary command to send a message to the APNS gateway (Internal use)
     */
    const BINARY_COMMAND = 1;

    /**
     * Binary size of a device token (Internal use)
     */
    const BINARY_DEVICE_TOKEN_SIZE = 32;

    /**
     * @var ConnectionHandler
     */
    private $connectionHandler;

    /**
     * AppleNotifier constructor.
     * @param ConnectionHandler $connectionHandler
     */
    public function __construct(ConnectionHandler $connectionHandler)
    {
        $this->connectionHandler = $connectionHandler;
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        parent::setLogger($logger);
        $this->connectionHandler->setLogger($logger);
    }

    /**
     * @param Message $message
     * @return bool
     */
    public function handles(Message $message): bool
    {
        return $message instanceof AppleMessage;
    }

    /**
     * @param Message $message
     * @return Response
     * @throws NotifyFailedException
     */
    protected function sendMessage(Message $message): Response
    {
        /* @var AppleMessage $message */
        try {
            $connection = $this->connectionHandler->getConnection($message->getConnectionType());

            $jsonMessage = json_encode($message->getMessage());
            $jsonMessageLength = strlen($jsonMessage);

            $payload =
                pack(
                    'CNNnH*n',
                    self::BINARY_COMMAND,
                    'aasdf',
                    $message->getExpiresAt(),
                    self::BINARY_DEVICE_TOKEN_SIZE,
                    $message->getDeviceToken(),
                    $jsonMessageLength
                );
            $payload .= $jsonMessage;
            return $connection->send($payload);
        } catch (Exception $e) {
            $this->getLogger()->error('Exception occurred sending an apple message: ' . $e->getMessage());
            throw new AppleNotifyFailedException($e->getMessage(), $e->getCode(), $e);
        }

    }

    /**
     * Get the message that this envelope contains in binary APNS compatible format
     *
     * @return string
     */
    public function getBinaryMessage()
    {
        $jsonMessage = $this->getMessage()->getJson();
        $jsonMessageLength = strlen($jsonMessage);

        $binaryMessage = pack('CNNnH*', self::BINARY_COMMAND, $this->getIdentifier(), $this->getMessage()->getExpiresAt(),
                self::BINARY_DEVICETOKEN_SIZE, $this->getMessage()->getDeviceToken()) . pack('n', $jsonMessageLength);
        return $binaryMessage . $jsonMessage;
    }
}