<?php

declare(strict_types=1);

namespace IssetBV\Presenter;

use IssetBV\Presenter\Exception\PresenterException;
use IssetBV\Util\Optional;
use PhpOption\Option;
use Symfony\Component\PropertyAccess\Exception\AccessException;
use Symfony\Component\PropertyAccess\Exception\InvalidArgumentException;
use Symfony\Component\PropertyAccess\Exception\UnexpectedTypeException;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Component\PropertyAccess\PropertyAccessor;
use Traversable;

/**
 * Class BasePresenter.
 *
 * @todo add advanced key sorting with closures
 *
 * @author Tim Fennis <tim@isset.nl>
 * @author Felix Balfoort <felix@isset.nl>
 */
abstract class PresenterAdapter implements Presenter
{
    const PRESENTER_INLINE = 1;

    /**
     * Child presenter mapping.
     *
     * @var array
     */
    private $childPresenters;

    /**
     * @var array
     */
    private $emptyValues;

    /**
     * @var bool
     */
    private $inline;

    /**
     * @var callable[]
     */
    private $closures;

    /**
     * @var callable
     */
    private $keyConverter;

    /**
     * @var PropertyAccessor
     */
    private $propertyAccessor;

    /**
     * PresenterAdapter constructor.
     *
     * @param int $options
     */
    public function __construct(int $options = 0)
    {
        $this->propertyAccessor = PropertyAccess::createPropertyAccessor();
        $this->childPresenters = [];
        $this->closures = [];
        $this->inline = (bool) ($options & self::PRESENTER_INLINE);
        $this->emptyValues = [];

        // add other options below
    }

    /**
     * @param $value
     */
    public function supportEmptyValue($value)
    {
        $this->emptyValues[] = $value;
    }

    /**
     * @param mixed $presentable
     *
     * @throws PresenterException
     *
     * @return array|int|string
     */
    final public function present($presentable)
    {
        if (true === is_array($presentable) || $presentable instanceof Traversable) {
            return \Functional\map($presentable, function ($element) {
                return $this->presentSingleInternal($element);
            });
        } else {
            return $this->presentSingleInternal($presentable);
        }
    }

    /**
     * @param callable $keyConverter
     */
    public function setKeyConverter(callable $keyConverter)
    {
        $this->keyConverter = $keyConverter;
        // TODO: Implement setKeyConverter() method.
    }

    /**
     * Map a presenter to a child property.
     *
     * @param string $childName
     * @param Presenter $presenter
     * @param string $overrideKey
     */
    public function mapChildPresenter(string $childName, Presenter $presenter, string $overrideKey = null)
    {
        $this->childPresenters[$childName] = [
            'presenter' => $presenter,
            'overrideKey' => $overrideKey,
        ];
    }

    /**
     * @param callable $closure
     */
    public function registerClosure(callable $closure)
    {
        $this->closures[] = $closure;
    }

    /**
     * Type hint the param annotation in subclass to get auto-complete.
     *
     * @param mixed $presentable Whatever needs to be presented
     *
     * @return array
     */
    abstract protected function presentSingle($presentable);

    /**
     * @param mixed $presentable
     *
     * @return mixed
     */
    abstract protected function presentSingleInline($presentable);

    /**
     * If you pass this presenter an optional it will automatically be unpacked.
     *
     * @param mixed $presentable
     *
     * @throws PresenterException
     *
     * @return array|mixed
     */
    final private function presentSingleInternal($presentable)
    {
        // Unpack
        if ($presentable instanceof Optional) {
            $presentable = $presentable->orElse(null);

            if ($presentable === null) {
                return null; //@todo not sure if it's the best idea to return directly
            }
        }

        if ($presentable instanceof Option) {
            $presentable = $presentable->getOrElse(null);

            if ($presentable === null) {
                return null; //@todo not sure if it's the best idea to return directly
            }
        }

        if (in_array($presentable, $this->emptyValues, true)) {
            return $this->presentEmptyValue();
        } else if ($this->supports($presentable)) {
            if ($this->inline === true) {
                $data = $this->presentSingleInline($presentable);
            } else {
                $data = $this->presentSingle($presentable);

                $this->executeMappedChildPresenters($data, $presentable);
                $this->executeClosures($data, $presentable);
                $this->formatDateTimeObjects($data);
            }
        } else {
            throw PresenterException::createTypeException($presentable, $this);
        }

        return $data;
    }

    /**
     * This function contains a lot of magic but makes your life a lot easier. As long as your presenter is mapped to a property that has a getter on the object.
     *
     * @param array $data
     * @param mixed $presentable
     *
     * @throws PresenterException
     */
    final private function executeMappedChildPresenters(array &$data, $presentable)
    {
        try {
            /* @var $presenter Presenter */
            foreach ($this->childPresenters as $key => $presenterData) {
                $response = $this->propertyAccessor->getValue($presentable, $key);

                /** @var Presenter $presenter */
                $presenter = $presenterData['presenter'];
                $dataKey = $presenterData['overrideKey'] ?: $key;

                $data[$dataKey] = $presenter->present($response);
            }
        } catch (AccessException $e) {
            throw new PresenterException($e->getMessage());
        } catch (InvalidArgumentException $e) {
            throw new PresenterException($e->getMessage());
        } catch (UnexpectedTypeException $e) {
            throw new PresenterException($e->getMessage());
        }
    }

    /**
     * @param array $data The data array to append the hooks data to
     * @param mixed $object The object used by the hooks
     */
    final private function executeClosures(array &$data, $object)
    {
        foreach ($this->closures as $lambda) {
            foreach ($lambda($object) as $key => $item) {
                $data[$key] = $item;
            }
        }
    }

    /**
     * override this in subclass.
     *
     * @return mixed
     */
    private function presentEmptyValue()
    {
        // default behavior is to return null
        return null;
    }

    /**
     * Search for values in $data of type DateTime and format them to an ISO8601 compatiable string
     *
     * @param array $data
     */
    private function formatDateTimeObjects(array &$data)
    {
        foreach ($data as $key => $value) {
            if ($value instanceof \DateTimeInterface) {
                $data[$key] = $value->format(\DateTime::W3C);
            }
        }
    }

}
