<?php

declare(strict_types=1);

namespace IssetBV\Presenter;

use DateTimeInterface;
use IssetBV\Presenter\Exception\PresenterException;

/**
 * Class BasePresenter.
 *
 * @author Tim Fennis <tim@isset.nl>
 *
 * @deprecated This class only provides some buggy compatibility layer
 */
abstract class BasePresenter extends PresenterAdapter
{
    /**
     * @var array
     */
    protected $childPresenters = [];

    /**
     * @var bool
     */
    protected $injectKeyConverterInChildPresenters;

    /**
     * @var bool
     */
    protected $inline = false;

    /**
     * @var callable
     */
    protected $keyConverter = null;

    /**
     * @var array
     */
    protected $closures = [];

    /**
     * @var bool
     */
    protected $restoreKeyConverterInChildPresenters;

    /**
     * @var bool
     */
    protected $sort = false;

    /**
     * @var Presenter[]
     */
    protected $typePresenters = [];

    /**
     * @param callable $closure
     */
    public function registerClosure(callable $closure)
    {
        $this->closures[] = $closure;
    }

    /**
     * If true the array keys will be sorted using ksort.
     *
     * @param bool $sort
     */
    public function setSort($sort)
    {
        $this->sort = $sort;
    }

    /**
     * Give this presenter a new KeyConverter to convert keys.
     *
     * @param callable $keyConverter
     */
    public function setKeyConverter(callable $keyConverter)
    {
        $this->keyConverter = $keyConverter;
    }

    /**
     * @return callable
     */
    public function getKeyConverter()
    {
        return $this->keyConverter;
    }

    /**
     * @param bool $inject
     */
    public function setInjectKeyConverterInChildPresenters($inject)
    {
        $this->injectKeyConverterInChildPresenters = $inject;
    }

    /**
     * @param bool $restore
     */
    public function setRestoreKeyConverterInChildPresenters($restore)
    {
        $this->restoreKeyConverterInChildPresenters = $restore;
    }

    /**
     * Map a presenter to a child property.
     *
     * @param string $childName
     * @param Presenter $presenter
     * @param string $overrideKey
     */
    public function mapChildPresenter(string $childName, Presenter $presenter, string $overrideKey = null)
    {
        $this->childPresenters[$childName] = [
            'presenter' => $presenter,
            'overrideKey' => $overrideKey,
        ];
    }

    /**
     * @param $fqnClassName
     * @param Presenter $presenter
     */
    public function mapTypePresenter($fqnClassName, Presenter $presenter)
    {
        $this->typePresenters[$fqnClassName] = $presenter;
    }

    /**
     * Tell this presenter to inline it's value.
     *
     * @param bool $inline
     */
    public function setInline($inline)
    {
        $this->inline = $inline;
    }

    /**
     * @param array $data The data array to append the hooks data to
     * @param mixed $object The object used by the hooks
     *
     * @return array
     */
    protected function executePresenterHooks(array &$data, $object)
    {
        /** @var callable $lambda */
        foreach ($this->closures as $lambda) {
            foreach ($lambda($object) as $item) {
                $data[] = $item;
            }
        }
    }

    /**
     * Overwrite this function to provide the implementation of this presenter (bread and butter here!).
     *
     * @param mixed $object
     *
     * @throws PresenterException
     *
     * @return string|int|array
     */
    abstract protected function presentSingleImpl($object);

    /**
     * Overwrite this function and make it provide an inline value.
     *
     * @param mixed $object
     *
     * @deprecated Replaced with presentSingleInline
     *
     * @return string|int|float
     */
    protected function presentInlineValue($object)
    {
        return (string) $object;
    }

    /**
     * Overwrite this function and make it provide a value for when the object is empty.
     *
     * @return mixed
     */
    protected function presentEmptyValue()
    {
        return null;
    }

    /**
     * @param mixed $object
     *
     * @throws PresenterException
     *
     * @deprecated use present instead (Or use PresenterAdapter)
     *
     * @return string|int|array
     */
    public function presentSingle($object)
    {
        if ($object === null) {
            return $this->presentEmptyValue();
        }

        if ($this->inline === true) {
            return $this->presentSingleInline($object);
        }

        $data = $this->present($object);

        // Sort the array keys alphabetically if requested
        if ($this->sort === true) {
            ksort($data);
        }

        // Convert all the keys using a keyConverter (if available)
        if ($this->keyConverter !== null) {
            $newData = [];
            foreach ($data as $key => $value) {
                $converter = $this->keyConverter;
                $newData[$converter($key)] = $value;
            }
            $data = $newData;
        }

        // Execute type presenters
        $classKey = get_class($object);
        if (true === array_key_exists($classKey, $this->typePresenters)) {
            $presenter = $this->typePresenters[$classKey];
            $typeData = $presenter->present($object);
            $data = array_merge($typeData, $data);
        }

        return $data;
    }

    /**
     * @param DateTimeInterface $dateTime
     * @param mixed $or
     *
     * @return string|mixed
     */
    public function dateOr(DateTimeInterface $dateTime = null, $or = null)
    {
        if ($dateTime === null) {
            return $or;
        } else {
            return $dateTime->format(\DateTime::W3C);
        }
    }

    /**
     * @param mixed $presentable
     *
     * @return bool
     */
    public function supports($presentable) : bool
    {
        return true;
    }

    /**
     * @param mixed $presentable
     *
     * @return mixed
     */
    protected function presentSingleInline($presentable)
    {
        return $this->presentInlineValue($presentable);
    }
}
