<?php

namespace IssetBV\Core\PresenterBundle\Presenter;

use Closure;
use DateTimeInterface;
use IssetBV\Core\PresenterBundle\Converter\KeyConverterInterface;
use IssetBV\Core\PresenterBundle\Exception\PresenterException;
use Traversable;

/**
 * Class BasePresenter
 *
 * @todo add advanced key sorting with closures
 *
 * @author Tim Fennis <tim@isset.nl>
 * @author Felix Balfoort <felix@isset.nl>
 */
abstract class BasePresenter implements PresenterInterface
{

    /**
     * @var array
     */
    protected $childPresenters = array();

    /**
     * @var boolean
     */
    protected $injectKeyConverterInChildPresenters;

    /**
     * @var boolean
     */
    protected $inline = false;

    /**
     * @var Closure
     */
    protected $keyConverter = null;

    /**
     * @var array
     */
    protected $presenterHooks = array();

    /**
     * @var boolean
     */
    protected $restoreKeyConverterInChildPresenters;

    /**
     * @var boolean
     */
    protected $sort = false;

    /**
     * @var PresenterInterface[]
     */
    protected $typePresenters = array();

    /**
     * @param Closure $closure
     */
    public function addPresenterHook(Closure $closure)
    {
        $this->presenterHooks[] = $closure;
    }

    /**
     * If true the array keys will be sorted using ksort
     *
     * @param boolean $sort
     */
    public function setSort($sort)
    {
        $this->sort = $sort;
    }

    /**
     * Give this presenter a new KeyConverter to convert keys
     *
     * @param Closure $keyConverter
     */
    public function setKeyConverter(Closure $keyConverter)
    {
        $this->keyConverter = $keyConverter;
    }

    /**
     * @return Closure
     */
    public function getKeyConverter()
    {
        return $this->keyConverter;
    }

    /**
     * @param boolean $inject
     */
    public function setInjectKeyConverterInChildPresenters($inject)
    {
        $this->injectKeyConverterInChildPresenters = $inject;
    }

    /**
     * @param boolean $restore
     */
    public function setRestoreKeyConverterInChildPresenters($restore)
    {
        $this->restoreKeyConverterInChildPresenters = $restore;
    }

    /**
     * Map a presenter to a child property
     *
     * @param string $childName
     * @param PresenterInterface $presenter
     * @param string $overrideKey
     * @return void
     */
    public function mapChildPresenter($childName, PresenterInterface $presenter, $overrideKey = null)
    {
        $this->childPresenters[$childName] = array(
            'presenter' => $presenter,
            'overrideKey' => $overrideKey
        );
    }

    /**
     * @param $fqnClassName
     * @param PresenterInterface $presenter
     */
    public function mapTypePresenter($fqnClassName, PresenterInterface $presenter)
    {
        $this->typePresenters[$fqnClassName] = $presenter;
    }

    /**
     * Converts a collection of data to an encodable array or value
     *
     * @param Traversable|array $collection
     * @return mixed
     * @throws PresenterException
     */
    public function present($collection)
    {
        $output = array();

        foreach ($collection as $partner) {
            $output[] = $this->presentSingle($partner);
        }

        return $output;
    }

    /**
     * Tell this presenter to inline it's value
     *
     * @param boolean $inline
     * @return void
     */
    public function setInline($inline)
    {
        $this->inline = $inline;
    }

    /**
     * This function contains a lot of magic but makes your life a lot easier. As long as your presenter is mapped to a property that has a getter on the object
     *
     * @param array $data
     * @param $object
     * @return array
     */
    protected function executeMappedChildPresenters(array $data, $object)
    {
        /** @var $presenter PresenterInterface */
        foreach ($this->childPresenters as $key => $presenterData) {
            $guessedMethodName = 'get' . ucfirst($key);
            if (true === method_exists($object, $guessedMethodName)) {
                $response = $object->$guessedMethodName();
                $presenter = $presenterData['presenter'];
                $dataKey = $presenterData['overrideKey'] !== null ? $presenterData['overrideKey'] : $key;

                // Create a backup of the key converter of the child
                $childKeyConverterBackup = $presenter->getKeyConverter();

                if ($this->injectKeyConverterInChildPresenters === true) {
                    $presenter->setKeyConverter($this->getKeyConverter());
                }

                if ($this->keyConverter !== null) {
                    $converter = $this->keyConverter;
                    $dataKey = $converter($dataKey);
                }

                if (is_array($response) || $response instanceof Traversable) {
                    $data[$dataKey] = $presenter->present($response);
                } else {
                    $data[$dataKey] = $presenter->presentSingle($response);
                }

                if ($this->restoreKeyConverterInChildPresenters === true) {
                    $presenter->setKeyConverter($childKeyConverterBackup);
                }
            }
        }

        return $data;
    }

    /**
     * @param array $data The data array to append the hooks data to
     * @param mixed $object The object used by the hooks
     * @return array
     */
    protected function executePresenterHooks(array $data, $object)
    {
        /** @var Closure $closure */
        foreach ($this->presenterHooks as $closure) {
            $data = array_merge($data, $closure($object));
        }

        return $data;
    }

    /**
     * Overwrite this function to provide the implementation of this presenter (bread and butter here!)
     *
     * @param mixed $object
     * @return string|int|array
     * @throws PresenterException
     */
    protected function presentSingleImpl($object)
    {
        // Implement in subclass
    }

    /**
     * Overwrite this function and make it provide an inline value
     *
     * @param mixed $object
     * @return string|int|double
     */
    protected function presentInlineValue($object)
    {
        return $object->__toString();
    }

    /**
     * Overwrite this function and make it provide a value for when the object is empty
     *
     * @return mixed
     */
    protected function presentEmptyValue()
    {
        return null;
    }

    /**
     * @param mixed $object
     * @return string|int|array
     * @throws PresenterException
     */
    public function presentSingle($object)
    {
        if ($object === null) {
            return $this->presentEmptyValue();
        }

        if ($this->inline === true) {
            return $this->presentInlineValue($object);
        }

        $data = $this->presentSingleImpl($object);

        $data = $this->executeMappedChildPresenters($data, $object);
        $data = $this->executePresenterHooks($data, $object);

        // Sort the array keys alphabetically if requested
        if ($this->sort === true) ksort($data);

        // Convert all the keys using a keyConverter (if available)
        if ($this->keyConverter !== null) {
            $newData = array();
            foreach ($data as $key => $value) {
                $converter = $this->keyConverter;
                $newData[$converter($key)] = $value;
            }
            $data = $newData;
        }

        // Execute type presenters
        $classKey = get_class($object);
        if (true === isset($this->typePresenters[$classKey])) {
            $presenter = $this->typePresenters[$classKey];
            $typeData = $presenter->presentSingle($object);
            $data = array_merge($typeData, $data);
        }

        return $data;
    }

    /**
     * @param DateTimeInterface $dateTime
     * @param mixed $or
     * @return string|mixed
     */
    public function dateOr(DateTimeInterface $dateTime = null, $or = null)
    {
        if ($dateTime === null) {
            return $or;
        } else {
            return $dateTime->format(\DateTime::W3C);
        }
    }
}