<?php

declare (strict_types = 1);

namespace IssetBV\Json;

use ArrayIterator;
use Closure;
use IssetBV\Json\Exception\UnsupportedOperationException;

/**
 * Class JsonArray.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class JsonArray implements JsonThing
{
    /**
     * @var array
     */
    private $data;

    /**
     * JsonArray constructor.
     *
     * @param JsonThing[] $data
     */
    public function __construct($data)
    {
        $this->data = $data;
    }

    /**
     * {@inheritdoc}
     *
     * @return ArrayIterator
     */
    public function getIterator()
    {
        return new ArrayIterator(array_values($this->data));
    }

    /**
     * {@inheritdoc}
     *
     * @param int $offset
     *
     * @return bool
     */
    public function offsetExists($offset)
    {
        return array_key_exists((int) $offset, $this->data);
    }

    /**
     * {@inheritdoc}
     *
     * @param int $offset
     *
     * @return JsonThing
     */
    public function offsetGet($offset)
    {
        return $this->data[(int) $offset];
    }

    /**
     * {@inheritdoc}
     *
     * @param int $offset
     * @param JsonThing $value
     *
     * @throws UnsupportedOperationException
     */
    public function offsetSet($offset, $value)
    {
        throw new UnsupportedOperationException('JsonArray is immutable');
    }

    /**
     * @param int $offset
     *
     * @throws UnsupportedOperationException
     */
    public function offsetUnset($offset)
    {
        throw new UnsupportedOperationException('JsonArray is immutable');
    }

    /**
     * @param string $propertyName
     *
     * @return bool
     */
    public function hasProperty(string $propertyName) : bool
    {
        // Arrays can't have properties
        return false;
    }

    /**
     * @return int The amount of elements in this array
     */
    public function count()
    {
        return count($this->data);
    }

    /**
     * @param Closure $consumer
     *
     * @return array
     */
    public function map(Closure $consumer) : array
    {
        $result = [];
        foreach ($this->data as $element) {
            $result[] = $consumer($element);
        }

        return $result;
    }

    /**
     * @todo I wish phpstorm would allow me to call this function forEach
     *
     * @param Closure $consumer
     */
    public function forAll(Closure $consumer)
    {
        foreach ($this->data as $element) {
            $consumer($element);
        }
    }
}
