<?php

declare (strict_types = 1);

namespace IssetBV\Json;

use ArrayIterator;
use Closure;
use IssetBV\Json\Exception\InvalidPropertyAccessException;
use IssetBV\Json\Exception\UnsupportedOperationException;

/**
 * Class JsonObject.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class JsonObject implements JsonThing
{
    /**
     * @var array
     */
    protected $data;

    /**
     * JsonObject constructor.
     *
     * @param array $data
     */
    public function __construct(array $data = [])
    {
        $this->data = $data;
    }

    /**
     * @return string[]
     */
    public function getProperties()
    {
        return array_keys($this->data);
    }

    /**
     * @param string $name
     *
     * @throws InvalidPropertyAccessException
     *
     * @return JsonThing
     */
    public function __get($name)
    {
        if (true === array_key_exists($name, $this->data)) {
            return $this->data[$name];
        }

        throw new InvalidPropertyAccessException($name, $this);
    }

    /**
     * @param string $property
     * @param JsonThing $value
     *
     * @throws UnsupportedOperationException
     */
    public function __set($property, JsonThing $value)
    {
        throw new UnsupportedOperationException('JsonObjects are immutable');
    }


    public function __isset($name)
    {
        return $this->offsetExists($name);
    }

    /**
     * Retrieve an external iterator.
     *
     * @link http://php.net/manual/en/iteratoraggregate.getiterator.php
     *
     * @return ArrayIterator
     */
    public function getIterator()
    {
        return new ArrayIterator($this->data);
    }

    /**
     * @param string $propertyName
     *
     * @return bool
     */
    public function hasProperty(string $propertyName) : bool
    {
        return array_key_exists($propertyName, $this->data);
    }

    /**
     * @param string $offset
     *
     * @return bool
     */
    public function offsetExists($offset)
    {
        return $this->hasProperty($offset);
    }

    /**
     * @param string $offset
     *
     * @throws InvalidPropertyAccessException
     *
     * @return JsonThing|mixed
     */
    public function offsetGet($offset)
    {
        if (false === $this->hasProperty($offset)) {
            throw new InvalidPropertyAccessException($offset, $this);
        }

        return $this->data[$offset];
    }

    /**
     * @param string $offset
     * @param string $value
     *
     * @throws UnsupportedOperationException
     */
    public function offsetSet($offset, $value)
    {
        throw new UnsupportedOperationException('Json object is immutable');
    }

    /**
     * @param string $offset
     *
     * @throws UnsupportedOperationException
     */
    public function offsetUnset($offset)
    {
        throw new UnsupportedOperationException('Json object is immutable');// TODO: Implement offsetUnset() method.
    }

    /**
     * @return int The amount of properties on this object
     */
    public function count()
    {
        return count($this->data);
    }

    /**
     * @param Closure $consumer Must be a bi-consumer that accepts property and value
     */
    public function forAll(Closure $consumer)
    {
        foreach ($this->data as $property => $value) {
            $consumer($property, $value);
        }
    }
}
