<?php

namespace IssetBV\Json;
use IssetBV\Json\Exception\InvalidPropertyAccessException;
use IssetBV\Json\Exception\UnsupportedOperationException;
use Traversable;

/**
 * Class JsonObject
 *
 * @package IssetBV\Json
 * @author Tim Fennis <tim@isset.nl>
 */
class JsonObject implements JsonThing
{
    /**
     * @var JsonThing[]
     */
    protected $data;

    /**
     * JsonObject constructor.
     *
     * @param array $data
     */
    public function __construct(array $data = [])
    {
        $this->data = $data;
    }

    /**
     * @return string[]
     */
    public function getProperties()
    {
        return array_keys($this->data);
    }

    /**
     * @param string $name
     * @return JsonThing
     * @throws InvalidPropertyAccessException
     */
    public function __get($name)
    {

        if (true === array_key_exists($name, $this->data)) {
            return $this->data[$name];
        }

        throw new InvalidPropertyAccessException($name, $this);
    }

    /**
     * @param string $property
     * @param JsonThing $value
     *
     * @throws UnsupportedOperationException
     */
    public function __set($property, JsonThing $value)
    {
        throw new UnsupportedOperationException('JsonObjects are immutable');
    }

    /**
     * Retrieve an external iterator
     *
     * @link http://php.net/manual/en/iteratoraggregate.getiterator.php
     * @return Traversable An instance of an object implementing <b>Iterator</b> or
     * <b>Traversable</b>
     * @since 5.0.0
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->data);
    }

    /**
     * @param string $propertyName
     * @return bool
     */
    public function hasProperty($propertyName)
    {
        return array_key_exists($propertyName, $this->data);
    }
}