<?php

declare(strict_types=1);

namespace IssetBV\NginxACL;

use DateTime;

/**
 * Class NginxAcl.
 *
 * build string vars:
 * ::expires::
 * ::uri::
 * ::remote::
 * ::secret::
 */
class NginxACL
{
    /**
     * @var string
     */
    private $buildString;

    /**
     * @var string
     */
    private $secret;

    /**
     * @var string
     */
    private $queryToken;

    /**
     * @var string
     */
    private $queryExpires;

    public function __construct(
        string $secret,
        string $buildString = '::expires::::uri:: ::secret::',
        string $queryToken = 'secure_token',
        string $queryExpires = 'secure_expires'
    ) {
        $this->buildString = $buildString;
        $this->secret = $secret;
        $this->queryExpires = $queryExpires;
        $this->queryToken = $queryToken;
    }

    public function secureUrl(string $url, DateTime $expires, string $remoteAddress = null): string
    {
        $data = parse_url($url);
        if (null === $remoteAddress) {
            $remoteAddress = $data['host'];
        }

        if (!array_key_exists('path', $data)) {
            throw new \LogicException('path not found in url');
        }

        $token = $this->generateToken($expires, $data['path'], $remoteAddress);

        return $this->rebuildUrl($expires, $data, $token);
    }

    public function generateToken(
        DateTime $expires,
        string $uri,
        string $remoteAddress = null
    ): string {
        $secure = str_replace([
            '::expires::',
            '::uri::',
            '::remote::',
            '::secret::',
        ], [
            $expires->getTimestamp(),
            $uri,
            $remoteAddress,
            $this->secret,
        ], $this->buildString);

        $token = base64_encode($this->md5bin($secure));

        return str_replace([
            '+',
            '/',
            '=',
        ], [
            '-',
            '_',
            '',
        ], $token);
    }

    public function md5bin($target): string
    {
        $md5 = md5($target);
        $ret = '';

        for ($i = 0; $i < 32; $i += 2) {
            $ret .= \chr(hexdec($md5[$i + 1]) + hexdec($md5[$i]) * 16);
        }

        return $ret;
    }

    private function rebuildUrl(DateTime $expires, $urlData, string $token): string
    {
        $url = '';
        if (array_key_exists('scheme', $urlData)) {
            $url .= $urlData['scheme'] . '://';
        }

        if (array_key_exists('host', $urlData)) {
            $url .= $urlData['host'];
        }

        $url .= $urlData['path'] .= '?';

        if (array_key_exists('query', $urlData)) {
            $url .= $urlData['query'] . '&';
        }
        $url .= $this->queryToken . '=' . $token . '&' . $this->queryExpires . '=' . $expires->getTimestamp();

        if (array_key_exists('fragment', $urlData)) {
            $url .= '#' . $urlData['fragment'];
        }

        return $url;
    }
}
