<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\PlatformBundle\Service\Client;

use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\MyIsset\CoreBundle\Entity\Division;
use IssetBV\MyIsset\CoreBundle\Entity\User;
use IssetBV\MyIsset\PlatformBundle\Entity\Token;
use IssetBV\MyIsset\PlatformBundle\Repository\TokenRepository;

class TokenClient
{
    const PLATFORM_HATCHERY = 'hatchery';
    const PLATFORM_VIDEO_ARCHIVE = 'video-archive';
    const PLATFORM_CONVERTER = 'converter';
    const PLATFORM_PUBLISHER = 'publisher';
    const PLATFORM_STREAM = 'video-stream';

    /**
     * @var PlatformClient
     */
    private $platformClient;
    /**
     * @var TokenRepository
     */
    private $tokenRepository;
    /**
     * @var ObjectManager
     */
    private $objectManager;

    public function __construct(ObjectManager $objectManager, PlatformClient $platformClient, TokenRepository $tokenRepository)
    {
        $this->platformClient = $platformClient;
        $this->tokenRepository = $tokenRepository;
        $this->objectManager = $objectManager;
    }

    /**
     * @param User $user
     * @param Division $division
     * @param string $platform
     *
     * @throws PlatformClientException
     *
     * @return TokenInterface
     */
    public function getApiKeyFor(User $user, Division $division, string $platform): TokenInterface
    {
        $token = $this->tokenRepository->fetchFor($user, $division, $platform);
        if (null === $token) {
            $response = $this->platformClient->call(
                'GET',
                'api/platform/token/api/key',
                [
                    'x-account-uuid' => $user->getUuid(),
                    'x-division-uuid' => $division->getUuid(),
                    'x-platform' => $platform,
                ]
            );

            if (200 !== $response->getStatusCode()) {
                throw PlatformClientException::create('Account not found', $response);
            }

            $token = new Token($user, $division, $platform, json_decode((string) $response->getBody())->token);
            $this->objectManager->persist($token);
            $this->objectManager->flush();
        }

        return $token;
    }
}
