<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\PlatformBundle\Service\Client;

use IssetBV\MyIsset\PlatformBundle\Template\Remote\Remote;
use IssetBV\MyIsset\PlatformBundle\Template\Remote\RemoteData;
use IssetBV\MyIsset\PlatformBundle\Template\Remote\RemoteDataException;
use Psr\Http\Message\ResponseInterface;

class RemotesClient
{
    /**
     * @var PlatformClient
     */
    private $platformClient;

    public function __construct(PlatformClient $platformClient)
    {
        $this->platformClient = $platformClient;
    }

    /**
     * @param string $accountUuid
     * @param string $divisionUuid
     *
     * @throws PlatformClientException
     * @throws RemoteDataException
     *
     * @return Remote[]
     */
    public function getRemotes(string $accountUuid, string $divisionUuid): array
    {
        $response = $this->platformClient->call('GET', 'api/platform/account/remotes', [
            'x-account-uuid' => $accountUuid,
            'x-division-uuid' => $divisionUuid,
        ]);

        if (200 !== $response->getStatusCode()) {
            throw PlatformClientException::create('Account not found', $response);
        }

        $remotes = json_decode((string) $response->getBody());

        return RemoteData::fromArray($remotes);
    }

    /**
     * @param string $accountUuid
     * @param Remote[] $remote
     *
     * @throws PlatformClientException
     *
     * @return ResponseInterface
     */
    public function createRemote(string $accountUuid, array $remote): ResponseInterface
    {
        $response = $this->platformClient->call('POST', 'api/platform/account/remotes', [
            'x-account-uuid' => $accountUuid,
        ], json_encode($remote));

        if (200 !== $response->getStatusCode()) {
            throw PlatformClientException::create('Remote not created', $response);
        }

        return $response;
    }

    /**
     * @param string $accountUuid
     * @param string $divisionUuid
     * @param int $remoteId
     *
     * @throws PlatformClientException
     * @throws RemoteDataException
     *
     * @return Remote
     */
    public function getRemote(string $accountUuid, string $divisionUuid, int $remoteId): Remote
    {
        $response = $this->platformClient->call('GET', 'api/platform/account/remotes/' . $remoteId, [
            'x-account-uuid' => $accountUuid,
            'x-division-uuid' => $divisionUuid,
        ]);

        if (200 !== $response->getStatusCode()) {
            throw PlatformClientException::create('Remote not found', $response);
        }

        $data = json_decode($response->getBody()->getContents());

        return RemoteData::fromDetails($remoteId, $data);
    }

    /**
     * @param string $accountUuid
     * @param Remote $remote
     *
     * @throws PlatformClientException
     *
     * @return ResponseInterface
     */
    public function patchRemote(string $accountUuid, Remote $remote): ResponseInterface
    {
        $response = $this->platformClient->call(
            'POST',
            'api/platform/account/remotes/' . $remote->getRemoteId(),
            [
                'x-account-uuid' => $accountUuid,
            ],
            json_encode($remote)
        );

        if (204 !== $response->getStatusCode()) {
            throw PlatformClientException::create('Remote not patched', $response);
        }

        return $response;
    }

    /**
     * @param string $accountUuid
     * @param int $remoteId
     *
     * @throws PlatformClientException
     */
    public function deleteRemote(string $accountUuid, int $remoteId)
    {
        $response = $this->platformClient->call(
            'DELETE',
            'api/platform/account/remotes/' . $remoteId,
            [
                'x-account-uuid' => $accountUuid,
            ]
        );

        if (204 !== $response->getStatusCode()) {
            throw PlatformClientException::create('Remote not deleted', $response);
        }
    }
}
