<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\PlatformBundle\Controller;

use IssetBV\MyIsset\CoreBundle\Service\Security\User;
use IssetBV\MyIsset\PlatformBundle\Service\Client\PlatformClientException;
use IssetBV\MyIsset\PlatformBundle\Service\Client\RemotesClient;
use IssetBV\MyIsset\PlatformBundle\Template\Remote\RemoteData;
use IssetBV\MyIsset\PlatformBundle\Template\Remote\RemoteDataException;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

class RemoteController extends Controller
{
    public function indexAction(RemotesClient $remotesClient): Response
    {
        $user = $this->getUser();
        /* @var User $user */

        try {
            $remotes = $remotesClient->getRemotes($user->getUser()->getUuid(), $user->getCurrentDivision()->getUuid());

            return new JsonResponse([
                'remotes' => $remotes,
            ]);
        } catch (PlatformClientException $e) {
            throw new BadRequestHttpException($e->getMessage());
        } catch (RemoteDataException $e) {
            throw new BadRequestHttpException($e->getMessage());
        }
    }

    public function createAction(Request $request, RemotesClient $remotesClient): Response
    {
        $user = $this->getUser();
        /* @var User $user */

        try {
            $response = $remotesClient->createRemote($user->getUser()->getUuid(), $request->request->all());

            return new JsonResponse(null, $response->getStatusCode());
        } catch (PlatformClientException $e) {
            throw new BadRequestHttpException($e->getMessage());
        }
    }

    public function detailAction(RemotesClient $remotesClient, int $remoteId): Response
    {
        $user = $this->getUser();
        /* @var User $user */

        try {
            $remote = $remotesClient->getRemote($user->getUser()->getUuid(), $user->getCurrentDivision()->getUuid(), $remoteId);

            return new JsonResponse([
                'remote' => $remote,
            ]);
        } catch (PlatformClientException $e) {
            throw new BadRequestHttpException($e->getMessage());
        } catch (RemoteDataException $e) {
            throw new BadRequestHttpException($e->getMessage());
        }
    }

    public function patchAction(Request $request, RemotesClient $remotesClient, int $remoteId): Response
    {
        $user = $this->getUser();
        /* @var User $user */

        try {
            $remote = RemoteData::fromDetails($remoteId, (object) $request->request->all());
            $response = $remotesClient->patchRemote($user->getUser()->getUuid(), $remote);

            return new JsonResponse(null, $response->getStatusCode());
        } catch (PlatformClientException $e) {
            throw new BadRequestHttpException($e->getMessage());
        } catch (RemoteDataException $e) {
            throw new BadRequestHttpException($e->getMessage());
        }
    }

    public function deleteAction(RemotesClient $remotesClient, int $remoteId): Response
    {
        $user = $this->getUser();
        /* @var User $user */

        try {
            $remotesClient->deleteRemote($user->getUser()->getUuid(), $remoteId);

            return new JsonResponse(null, 204);
        } catch (PlatformClientException $e) {
            throw new BadRequestHttpException($e->getMessage());
        }
    }
}
