<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\PlatformBundle\Service\Client;

use IssetBV\MyIsset\CoreBundle\Service\Client\Core\Client as CoreClient;
use IssetBV\MyIsset\CoreBundle\Service\Client\Core\Request;
use IssetBV\MyIsset\CoreBundle\Service\Client\Core\RequestException;
use IssetBV\MyIsset\PlatformBundle\Service\Data\Remote;
use IssetBV\MyIsset\PlatformBundle\Service\Data\RemoteData;
use Psr\Http\Message\ResponseInterface;
use RuntimeException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

class Remotes
{
    /**
     * @var CoreClient
     */
    private $client;
    /**
     * @var string
     */
    private $token;

    /**
     * @param CoreClient $client
     * @param string     $token
     */
    public function __construct(CoreClient $client, string $token)
    {
        $this->client = $client;
        $this->token = $token;
    }

    /**
     * @param string $accountUuid
     * @param string $divisionUuid
     *
     * @throws RequestException
     * @throws BadRequestHttpException
     * @throws RuntimeException
     *
     * @return Remote[]
     */
    public function getRemotes(string $accountUuid, string $divisionUuid)
    {
        $request = new Request('GET', 'api/platform/account/remotes', [
            'x-my-platform' => $this->token,
            'x-account-uuid' => $accountUuid,
            'x-division-uuid' => $divisionUuid,
        ]);

        try {
            $response = $this->client->call($request);
            $data = json_decode($response->getBody()->getContents());

            $remotes = [];
            foreach ($data as $remoteData) {
                $remotes[] = RemoteData::fromList($remoteData);
            }

            return $remotes;
        } catch (RequestException $e) {
            throw new BadRequestHttpException('Account not found');
        }
    }

    /**
     * @param string $accountUuid
     * @param array  $remote
     *
     * @throws BadRequestHttpException
     *
     * @return ResponseInterface
     */
    public function createRemote(string $accountUuid, $remote)
    {
        $request = new Request('POST', 'api/platform/account/remotes', [
            'x-my-platform' => $this->token,
            'x-account-uuid' => $accountUuid,
        ], json_encode($remote));

        try {
            return $this->client->call($request);
        } catch (RequestException $e) {
            throw new BadRequestHttpException('Remote not created');
        }
    }

    /**
     * @param string $accountUuid
     * @param string $divisionUuid
     * @param int    $remoteId
     *
     * @throws RequestException
     * @throws BadRequestHttpException
     *
     * @return Remote
     */
    public function getRemote(string $accountUuid, string $divisionUuid, int $remoteId)
    {
        $request = new Request('GET', 'api/platform/account/remotes/' . $remoteId, [
            'x-my-platform' => $this->token,
            'x-account-uuid' => $accountUuid,
            'x-division-uuid' => $divisionUuid,
        ]);

        try {
            $response = $this->client->call($request);
            $data = json_decode($response->getBody()->getContents());

            return RemoteData::fromDetails($remoteId, $data);
        } catch (RequestException $e) {
            throw new BadRequestHttpException('Remote not found');
        }
    }

    /**
     * @param string $accountUuid
     * @param Remote $remote
     *
     * @throws BadRequestHttpException
     *
     * @return ResponseInterface
     */
    public function patchRemote(string $accountUuid, Remote $remote)
    {
        $request = new Request('POST', 'api/platform/account/remotes/' . $remote->getRemoteId(), [
            'x-my-platform' => $this->token,
            'x-account-uuid' => $accountUuid,
        ], json_encode($remote));

        try {
            return $this->client->call($request);
        } catch (RequestException $e) {
            throw new BadRequestHttpException('Remote not patched');
        }
    }

    /**
     * @param int $remoteId
     *
     * @throws BadRequestHttpException
     *
     * @return ResponseInterface
     */
    public function deleteRemote(string $accountUuid, int $remoteId)
    {
        $request = new Request('DELETE', 'api/platform/account/remotes/' . $remoteId, [
            'x-my-platform' => $this->token,
            'x-account-uuid' => $accountUuid,
        ]);

        try {
            return $this->client->call($request);
        } catch (RequestException $e) {
            throw new BadRequestHttpException('Remote not patched');
        }
    }
}
