<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\PlatformBundle\Controller;

use InvalidArgumentException;
use IssetBV\MyIsset\CoreBundle\Service\Client\Core\RequestException;
use IssetBV\MyIsset\CoreBundle\Service\Security\User;
use IssetBV\MyIsset\PlatformBundle\Service\Data\RemoteData;
use LogicException;
use RuntimeException;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

class RemoteController extends Controller
{
    /**
     * @throws LogicException
     * @throws RuntimeException
     * @throws RequestException
     * @throws BadRequestHttpException
     *
     * @return JsonResponse
     */
    public function indexAction()
    {
        /* @var User $user */
        $user = $this->getUser();
        $remoteProvider = $this->get('isset_bv_my_isset_platform.remote.provider');

        $remotes = $remoteProvider->getRemotes($user->getUser()->getUuid(), $user->getCurrentDivision()->getUuid());

        return new JsonResponse([
            'remotes' => $remotes,
        ]);
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function createAction(Request $request)
    {
        /* @var User $user */
        $user = $this->getUser();
        $remoteProvider = $this->get('isset_bv_my_isset_platform.remote.provider');

        $response = $remoteProvider->createRemote($user->getUser()->getUuid(), (object) $request->request->all());

        return new Response('', $response->getStatusCode());
    }

    /**
     * @param int $remoteId
     *
     * @throws RequestException
     * @throws LogicException
     *
     * @return JsonResponse
     */
    public function detailAction(int $remoteId)
    {
        /* @var User $user */
        $user = $this->getUser();
        $remoteProvider = $this->get('isset_bv_my_isset_platform.remote.provider');

        $response = $remoteProvider->getRemote($user->getUser()->getUuid(), $user->getCurrentDivision()->getUuid(), $remoteId);

        return new JsonResponse([
            'remote' => $response,
        ]);
    }

    /**
     * @param Request $request
     * @param int     $remoteId
     *
     * @throws InvalidArgumentException
     * @throws BadRequestHttpException
     * @throws LogicException
     *
     * @return Response
     */
    public function patchAction(Request $request, int $remoteId)
    {
        $remote = RemoteData::fromDetails($remoteId, (object) $request->request->all());

        /* @var User $user */
        $user = $this->getUser();
        $remoteProvider = $this->get('isset_bv_my_isset_platform.remote.provider');

        $response = $remoteProvider->patchRemote($user->getUser()->getUuid(), $remote);

        return new Response('', $response->getStatusCode());
    }

    /**
     * @param int $remoteId
     *
     * @throws BadRequestHttpException
     * @throws LogicException
     * @throws InvalidArgumentException
     *
     * @return Response
     */
    public function deleteAction(int $remoteId)
    {
        /* @var User $user */
        $user = $this->getUser();
        $remoteProvider = $this->get('isset_bv_my_isset_platform.remote.provider');

        $response = $remoteProvider->deleteRemote($user->getUser()->getUuid(), $remoteId);

        return new Response('', $response->getStatusCode());
    }
}
