<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\OAuthBundle\Service\Security\RememberMe;

use IssetBV\MyIsset\CoreBundle\Service\Security\User;
use Nette\Utils\DateTime;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Http\RememberMe\AbstractRememberMeServices;

class TokenBasedRememberMeService extends AbstractRememberMeServices
{
    /**
     * @var TokenProvider
     */
    private $tokenProvider;

    public function __construct(
        array $userProviders = [],
        $secret = '',
        $providerKey = '',
        array $options = [],
        LoggerInterface $logger = null,
        TokenProvider $tokenProvider
    ) {
        parent::__construct($userProviders, $secret, $providerKey, $options, $logger);
        $this->tokenProvider = $tokenProvider;
    }

    /**
     * {@inheritdoc}
     */
    protected function processAutoLoginCookie(array $cookieParts, Request $request)
    {
        if (4 !== \count($cookieParts)) {
            throw new AuthenticationException('The cookie is invalid.');
        }

        list($class, $token, $expires, $hash) = $cookieParts;
        if (false === $token = base64_decode($token, true)) {
            throw new AuthenticationException('$username contains a character from outside the base64 alphabet.');
        }

        $rememberMeToken = $this->tokenProvider->fetchByToken($token);
        if (null === $rememberMeToken) {
            throw new AuthenticationException('The cookie\'s hash is invalid.');
        }

        if (true !== hash_equals($this->generateCookieHash($class, $token, $expires, $rememberMeToken->getApiKey()), $hash)) {
            throw new AuthenticationException('The cookie\'s hash is invalid.');
        }

        try {
            $user = $this->getUserProvider($class)->loadUserByUsername($rememberMeToken->getApiKey());
            /* @var User $user */
        } catch (\Exception $e) {
            if (!$e instanceof AuthenticationException) {
                $e = new AuthenticationException($e->getMessage(), $e->getCode(), $e);
            }

            throw $e;
        }

        if (!$user instanceof UserInterface) {
            throw new \RuntimeException(
                sprintf(
                    'The UserProviderInterface implementation must return an instance of UserInterface, but returned "%s".',
                    \get_class($user)
                )
            );
        }

        if ($expires < time()) {
            throw new AuthenticationException('The cookie has expired.');
        }

        return $user;
    }

    /**
     * {@inheritdoc}
     */
    protected function onLoginSuccess(Request $request, Response $response, TokenInterface $token)
    {
        $user = $token->getUser();
        /* @var User $user */

        $expires = time() + $this->options['lifetime'];
        $expiresDateTime = new DateTime();
        $expiresDateTime->setTimestamp($expires);

        $token = $this->tokenProvider->provideFor($user->getUserToken(), $expiresDateTime);
        $value = $this->generateCookieValue(get_class($user), $token->getToken(), $expires, $user->getUserToken());

        $response->headers->setCookie(
            new Cookie(
                $this->options['name'],
                $value,
                $expires,
                $this->options['path'],
                $this->options['domain'],
                $this->options['secure'],
                $this->options['httponly']
            )
        );
    }

    /**
     * Generates the cookie value.
     *
     * @param string $class
     * @param string $username The username
     * @param int $expires The Unix timestamp when the cookie expires
     * @param string $password The encoded password
     *
     * @return string
     */
    protected function generateCookieValue($class, $username, $expires, $password)
    {
        // $username is encoded because it might contain COOKIE_DELIMITER,
        // we assume other values don't
        return $this->encodeCookie([
            $class,
            base64_encode($username),
            $expires,
            $this->generateCookieHash($class, $username, $expires, $password),
        ]);
    }

    /**
     * Generates a hash for the cookie to ensure it is not being tempered with.
     *
     * @param string $class
     * @param string $username The username
     * @param int $expires The Unix timestamp when the cookie expires
     * @param string $password The encoded password
     *
     * @return string
     */
    protected function generateCookieHash($class, $username, $expires, $password)
    {
        return hash_hmac('sha256', $class . $username . $expires . $password, $this->getSecret());
    }

    protected function isRememberMeRequested(Request $request)
    {
        switch ($request->get('remember_me')) {
            case '1':
            case true:
                return true;
            default:
                return false;
        }
    }
}
