<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\OAuthBundle\Service\Security\RememberMe;

use Symfony\Bundle\SecurityBundle\DependencyInjection\Security\Factory\SecurityFactoryInterface;
use Symfony\Component\Config\Definition\Builder\ArrayNodeDefinition;
use Symfony\Component\Config\Definition\Builder\NodeDefinition;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;

class RememberMeFactory implements SecurityFactoryInterface
{
    protected $options = [
        'name' => 'REMEMBERME',
        'lifetime' => 31536000,
        'path' => '/',
        'domain' => null,
        'secure' => false,
        'httponly' => true,
        'always_remember_me' => false,
        'remember_me_parameter' => '_remember_me',
    ];

    public function create(ContainerBuilder $container, $id, $config, $userProvider, $defaultEntryPoint)
    {
        // authentication provider
        $authProviderId = 'security.authentication.provider.rememberme.' . $id;
        $container
            ->setDefinition($authProviderId, new ChildDefinition('security.authentication.provider.rememberme'))
            ->replaceArgument(0, new Reference('security.user_checker.' . $id))
            ->addArgument($config['secret'])
            ->addArgument($id);

        $templateId = TokenBasedRememberMeService::class;
        $rememberMeServicesId = $templateId . '.' . $id;

        if ($container->hasDefinition('security.logout_listener.' . $id)) {
            $container
                ->getDefinition('security.logout_listener.' . $id)
                ->addMethodCall('addHandler', [new Reference($rememberMeServicesId)]);
        }

        $rememberMeServices = $container->setDefinition($rememberMeServicesId, new ChildDefinition($templateId));
        $rememberMeServices->replaceArgument(1, $config['secret']);
        $rememberMeServices->replaceArgument(2, $id);
        $rememberMeServices->replaceArgument('$tokenProvider', new Reference(TokenProvider::class));

        // remember-me options
        $rememberMeServices->replaceArgument(3, array_intersect_key($config, $this->options));

        // attach to remember-me aware listeners
        $userProviders = [];
        foreach ($container->findTaggedServiceIds('security.remember_me_aware') as $serviceId => $attributes) {
            foreach ($attributes as $attribute) {
                if (!isset($attribute['id']) || $attribute['id'] !== $id) {
                    continue;
                }

                if (!isset($attribute['provider'])) {
                    throw new \RuntimeException('Each "security.remember_me_aware" tag must have a provider attribute.');
                }

                $userProviders[] = new Reference($attribute['provider']);
                $container
                    ->getDefinition($serviceId)
                    ->addMethodCall('setRememberMeServices', [new Reference($rememberMeServicesId)]);
            }
        }

        if (0 === count($userProviders)) {
            throw new \RuntimeException('You must configure at least one remember-me aware listener (such as form-login) for each firewall that has remember-me enabled.');
        }

        $rememberMeServices->replaceArgument(0, array_unique($userProviders));

        // remember-me listener
        $listenerId = 'security.authentication.listener.rememberme.' . $id;
        $listener = $container->setDefinition($listenerId, new ChildDefinition('security.authentication.listener.rememberme'));
        $listener->replaceArgument(1, new Reference($rememberMeServicesId));
        $listener->replaceArgument(5, true);

        return [$authProviderId, $listenerId, $defaultEntryPoint];
    }

    public function getPosition()
    {
        return 'remember_me';
    }

    public function getKey()
    {
        return 'remember-me-oauth';
    }

    public function addConfiguration(NodeDefinition $node)
    {
        if (!$node instanceof ArrayNodeDefinition) {
            throw new \LogicException('no node found');
        }
        /* @var ArrayNodeDefinition $node */
        $builder = $node
            ->children();

        $builder->scalarNode('secret')->isRequired()->cannotBeEmpty()->end();

        foreach ($this->options as $name => $value) {
            if (\is_bool($value)) {
                $builder->booleanNode($name)->defaultValue($value);
            } else {
                $builder->scalarNode($name)->defaultValue($value);
            }
        }
    }

    private function addConfigurationArray(ArrayNodeDefinition $node)
    {
        $builder = $node
            ->children();

        $builder->scalarNode('secret')->isRequired()->cannotBeEmpty()->end();

        foreach ($this->options as $name => $value) {
            if (\is_bool($value)) {
                $builder->booleanNode($name)->defaultValue($value);
            } else {
                $builder->scalarNode($name)->defaultValue($value);
            }
        }
    }
}
