<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\OAuthBundle\Service\Security;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Authentication\SimplePreAuthenticatorInterface;

class UserAuthenticator implements SimplePreAuthenticatorInterface, AuthenticationSuccessHandlerInterface
{
    /**
     * @var string
     */
    private $tokenName;

    /**
     * @var UserProviderInterface
     */
    private $userProvider;

    /**
     * @var AuthenticationSuccessHandler
     */
    private $authenticationSuccessHandler;

    /**
     * UserAuthenticator constructor.
     *
     * @param string $tokenName
     * @param AuthenticationSuccessHandler $authenticationSuccessHandler
     */
    public function __construct(string $tokenName, AuthenticationSuccessHandler $authenticationSuccessHandler)
    {
        $this->tokenName = $tokenName;
        $this->authenticationSuccessHandler = $authenticationSuccessHandler;
    }

    public function authenticateToken(TokenInterface $token, UserProviderInterface $userProvider, $providerKey): UserToken
    {
        $apiKey = $token->getCredentials();
        $user = $userProvider->loadUserByUsername($apiKey);
        $return = new UserToken($user, $apiKey, $providerKey, $user->getRoles());
        $return->setAuthenticated(true);
        $this->userProvider = $userProvider;

        return $return;
    }

    /**
     * @param Request $request
     * @param mixed $providerKey
     *
     * @throws \InvalidArgumentException
     * @throws \Symfony\Component\Security\Core\Exception\BadCredentialsException
     *
     * @return UserToken
     */
    public function createToken(Request $request, $providerKey): UserToken
    {
        $apiKey = $request->query->get($this->tokenName);
        if (!$apiKey) {
            throw new BadCredentialsException('No API key found');
        }

        return new UserToken('anon.', $apiKey, $providerKey);
    }

    /**
     * @param TokenInterface $token
     * @param mixed $providerKey
     *
     * @return bool
     */
    public function supportsToken(TokenInterface $token, $providerKey)
    {
        return $token instanceof UserToken && $token->getProviderKey() === $providerKey;
    }

    /**
     * This is called when an interactive authentication attempt succeeds. This
     * is called by authentication listeners inheriting from
     * AbstractAuthenticationListener.
     *
     * @param Request $request
     * @param TokenInterface $token
     *
     * @throws \InvalidArgumentException
     *
     * @return Response never null
     */
    public function onAuthenticationSuccess(Request $request, TokenInterface $token)
    {
        return $this->authenticationSuccessHandler->handle($request, $token, $this->userProvider);
    }
}
