<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\OAuthBundle\Service\Client;

use IssetBV\Http\Client\ClientInterface;
use IssetBV\Http\Client\Exception\ResponseException;
use IssetBV\MyIsset\CoreBundle\Service\Security\ClientData\ClientData;
use IssetBV\MyIsset\CoreBundle\Service\Security\ClientData\ClientDataGenerator;
use IssetBV\MyIsset\CoreBundle\Service\Security\ClientDataInterface;
use IssetBV\MyIsset\CoreBundle\Service\Security\Exception\AccountNotFound;
use IssetBV\MyIsset\CoreBundle\Service\Security\UserProviderClientInterface;
use IssetBV\MyIsset\OAuthBundle\Service\Client\Data\LoginToken;
use IssetBV\MyIsset\OAuthBundle\Service\Client\Exception\TokenNotCreatedException;

class Client implements UserProviderClientInterface
{
    /**
     * @var ClientInterface
     */
    private $client;
    /**
     * @var OAuthUrlProvider
     */
    private $OAuthUrlProvider;
    /**
     * @var ClientDataGenerator
     */
    private $clientDataGenerator;

    public function __construct(ClientInterface $client, OAuthUrlProvider $OAuthUrlProvider, ClientDataGenerator $clientDataGenerator)
    {
        $this->client = $client;
        $this->OAuthUrlProvider = $OAuthUrlProvider;
        $this->clientDataGenerator = $clientDataGenerator;
    }

    /**
     * @param string $token
     *
     * @throws AccountNotFound
     * @throws ResponseException
     *
     * @return ClientData
     */
    public function getAccount(string $token): ClientDataInterface
    {
        $url = $this->OAuthUrlProvider->createAccountUrl();
        $request = $this->client->getRequestGenerator()->create('GET', $url, ['x-mijn-isset' => $token]);
        $response = $this->client->call($request);
        if (200 !== $response->getStatusCode()) {
            throw AccountNotFound::create('Account not found', $response);
        }

        return $this->clientDataGenerator->fromResponse($response);
    }

    /**
     * @param string $publicKey
     * @param string $time
     * @param string $hash
     *
     * @throws TokenNotCreatedException
     *
     * @return LoginToken
     */
    public function getLoginToken(string $publicKey, string $time, string $hash): LoginToken
    {
        $url = $this->OAuthUrlProvider->createLoginTokenUrl($publicKey, $time, $hash);

        $request = $this->client->getRequestGenerator()->create('GET', $url);
        $response = $this->client->call($request);
        if (200 !== $response->getStatusCode()) {
            throw TokenNotCreatedException::create('token not found', $response);
        }

        $data = json_decode($response->getBody()->getContents());

        if (!isset($data->login_token)) {
            throw TokenNotCreatedException::create('token not found', $response);
        }

        return new LoginToken($data->login_token);
    }
}
