<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\CoreBundle\Service\Security;

use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\MyIsset\CoreBundle\Entity\User as UserEntity;
use IssetBV\MyIsset\CoreBundle\Service\Data\CompanyProviderInterface;
use IssetBV\MyIsset\CoreBundle\Service\Data\DivisionProviderInterface;
use IssetBV\MyIsset\CoreBundle\Service\Data\Exception\UserNotFound;
use IssetBV\MyIsset\CoreBundle\Service\Data\UserProviderInterface;
use IssetBV\MyIsset\CoreBundle\Service\Security\Exception\AccountNotFound;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface as SymfonyUserProviderInterface;

abstract class UserProvider implements SymfonyUserProviderInterface
{
    /**
     * @var ObjectManager
     */
    private $om;
    /**
     * @var EventDispatcherInterface
     */
    private $dispatcher;
    /**
     * @var UserProviderClientInterface
     */
    private $userProviderClient;
    /**
     * @var UserProviderInterface
     */
    private $userDataProvider;
    /**
     * @var CompanyProviderInterface
     */
    private $companyProvider;
    /**
     * @var DivisionProviderInterface
     */
    private $divisionProvider;

    public function __construct(
        ObjectManager $om,
        EventDispatcherInterface $dispatcher,
        UserProviderClientInterface $userProviderClient,
        UserProviderInterface $userDataProvider,
        CompanyProviderInterface $companyProvider,
        DivisionProviderInterface $divisionProvider
    ) {
        $this->om = $om;
        $this->dispatcher = $dispatcher;
        $this->userProviderClient = $userProviderClient;
        $this->userDataProvider = $userDataProvider;
        $this->companyProvider = $companyProvider;
        $this->divisionProvider = $divisionProvider;
    }

    /**
     * @param string $token
     *
     * @throws UnsupportedUserException
     *
     * @return UserInterface
     */
    public function loadUserByUsername($token)
    {
        try {
            $userData = $this->userProviderClient->getAccount($token);

            try {
                $user = $this->userDataProvider->fetchByUuid($userData->getUuid());
            } catch (UserNotFound $e) {
                $company = $this->companyProvider->fetchBy($userData->getOwner());
                $user = new UserEntity($company, $userData->getUuid(), $userData->getName(), $userData->getEmail());
                $this->om->persist($user);
            }
            $user->setRoles($userData->getRoles());

            foreach ($userData->getDivisions() as $divisionUuid) {
                if (!$user->hasDivisionUuid($divisionUuid->getUuid())) {
                    $user->addDivision($this->divisionProvider->fetchBy($divisionUuid));
                }
            }

            $user->clearDivisionsByKeep();

            $this->om->flush();
            $authUser = new User($user, $userData->getCurrentDivisionUuid(), $token);
            $this->dispatcher->dispatch(UserEvent::LOAD, new UserEvent($authUser));

            return $authUser;
        } catch (AccountNotFound $e) {
            throw new UnsupportedUserException();
        }
    }

    /**
     * Refreshes the user for the account interface.
     *
     * It is up to the implementation to decide if the user data should be
     * totally reloaded (e.g. from the database), or if the UserInterface
     * object can just be merged into some internal array of users / identity
     * map.
     *
     * @param UserInterface $authUser
     *
     * @throws UnsupportedUserException
     *
     * @return UserInterface
     */
    public function refreshUser(UserInterface $authUser)
    {
        if (!($authUser instanceof User)) {
            throw new UnsupportedUserException();
        }
        try {
            $user = $this->userDataProvider->fetchById($authUser->getEntityId());
            $authUser->setUser($user);
            $this->dispatcher->dispatch(UserEvent::LOAD, new UserEvent($authUser));

            return $authUser;
        } catch (UserNotFound $e) {
            throw new UnsupportedUserException();
        }
    }

    /**
     * Whether this provider supports the given user class.
     *
     * @param string $class
     *
     * @return bool
     */
    public function supportsClass($class)
    {
        return User::class === $class;
    }
}
