<?php

declare(strict_types=1);

namespace IssetBV\MyIsset\CoreBundle\Entity;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity(repositoryClass="IssetBV\MyIsset\CoreBundle\Repository\UserRepository")
 * @ORM\Table(name="myisset_users")
 */
class User
{
    /**
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var Company
     * @ORM\ManyToOne(targetEntity="Company")
     * @ORM\JoinColumn(nullable=false)
     */
    private $company;

    /**
     * @var Division[]
     * @ORM\ManyToMany(targetEntity="Division")
     * @ORM\JoinTable(name="myisset_users_divisions",
     *      joinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="id", onDelete="cascade")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="division_id", referencedColumnName="id", onDelete="cascade")}
     *)
     */
    private $divisions;

    /**
     * @var string
     * @ORM\Column(type="string", nullable=false, unique=true)
     */
    private $uuid;

    /**
     * @var string
     * @ORM\Column(type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     * @ORM\Column(type="string", nullable=false)
     */
    private $email;

    /**
     * @var array
     * @ORM\Column(type="json_array", nullable=false)
     */
    private $roles;

    /**
     * @var DateTime
     * @ORM\Column( type="datetime", nullable=false)
     */
    private $dateCreated;

    /**
     * @var DateTime
     * @ORM\Column(type="datetime", nullable=false)
     */
    private $dateUpdated;
    /**
     * @var Division[]
     */
    private $keep = [];

    /**
     * User constructor.
     *
     * @param Company $company
     * @param string $uuid
     * @param string $name
     * @param string $email
     */
    public function __construct(Company $company, string $uuid, string $name, string $email)
    {
        $this->company = $company;
        $this->uuid = $uuid;
        $this->name = $name;
        $this->email = $email;
        $this->roles = [];
        $this->divisions = new ArrayCollection();
        $this->dateCreated = new DateTime();
        $this->dateUpdated = new DateTime();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return Company
     */
    public function getCompany()
    {
        return $this->company;
    }

    /**
     * @return Division[]
     */
    public function getDivisions()
    {
        return $this->divisions;
    }

    /**
     * @return string
     */
    public function getUuid(): string
    {
        return $this->uuid;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getEmail(): string
    {
        return $this->email;
    }

    /**
     * @return array
     */
    public function getRoles(): array
    {
        return $this->roles;
    }

    /**
     * @return DateTime
     */
    public function getDateCreated(): DateTime
    {
        return $this->dateCreated;
    }

    /**
     * @return DateTime
     */
    public function getDateUpdated(): DateTime
    {
        return $this->dateUpdated;
    }

    /**
     * @param Division $division
     *
     * @return bool
     */
    public function hasDivision(Division $division)
    {
        foreach ($this->divisions as $divisionCheck) {
            if ($divisionCheck->getId() === $division->getId()) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param Division $division
     */
    public function addDivision(Division $division)
    {
        if (!$this->hasDivision($division)) {
            $this->divisions->add($division);
            $this->keep[] = $division;
        }
    }

    /**
     * @param array $roles
     */
    public function setRoles($roles)
    {
        $this->roles = $roles;
    }

    public function clearDivisionsByKeep()
    {
        foreach ($this->divisions as $division) {
            if (!\in_array($division, $this->keep)) {
                $this->divisions->removeElement($division);
            }
        }
        $this->keep = [];
    }

    public function hasDivisionUuid(string $divisionUuid, bool $tagAsKeep = true): bool
    {
        foreach ($this->divisions as $division) {
            if ($division->getUuid() === $divisionUuid) {
                if ($tagAsKeep && !\in_array($division, $this->keep)) {
                    $this->keep[] = $division;
                }

                return true;
            }
        }

        return false;
    }
}
