<?php
declare(strict_types = 1);

namespace IssetBV\MyIsset\CoreBundle\Service\Security;

use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\MyIsset\CoreBundle\Entity\User as UserEntity;
use IssetBV\MyIsset\CoreBundle\Service\Data\CompanyProvider;
use IssetBV\MyIsset\CoreBundle\Service\Data\DivisionProvider;
use IssetBV\MyIsset\CoreBundle\Service\Data\Exception\UserNotFound;
use IssetBV\MyIsset\CoreBundle\Service\Data\UserProvider as UserDataProvider;
use IssetBV\MyIsset\CoreBundle\Service\Security\Exception\AccountNotFound;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;

class UserProvider implements UserProviderInterface
{

    /**
     * @var ObjectManager
     */
    private $om;
    /**
     * @var EventDispatcherInterface
     */
    private $dispatcher;
    /**
     * @var UserProviderClient
     */
    private $userProviderClient;
    /**
     * @var UserDataProvider
     */
    private $userProvider;
    /**
     * @var CompanyProvider
     */
    private $companyProvider;
    /**
     * @var DivisionProvider
     */
    private $divisionProvider;

    /**
     * UserProvider constructor.
     * @param ObjectManager $om
     * @param EventDispatcherInterface $dispatcher
     * @param UserProviderClient $userProviderClient
     * @param UserDataProvider $userProvider
     * @param CompanyProvider $companyProvider
     * @param DivisionProvider $divisionProvider
     */
    public function __construct(
        ObjectManager $om,
        EventDispatcherInterface $dispatcher,
        UserProviderClient $userProviderClient,
        UserDataProvider $userProvider,
        CompanyProvider $companyProvider,
        DivisionProvider $divisionProvider
    ) {
        $this->om = $om;
        $this->dispatcher = $dispatcher;
        $this->userProviderClient = $userProviderClient;
        $this->userProvider = $userProvider;
        $this->companyProvider = $companyProvider;
        $this->divisionProvider = $divisionProvider;
    }

    /**
     * @param string $token
     * @return UserInterface|void
     * @throws \Symfony\Component\Security\Core\Exception\UnsupportedUserException
     */
    public function loadUserByUsername($token)
    {
        try {
            $userData = $this->userProviderClient->getAccount($token);

            try {
                $user = $this->userProvider->fetchByUuid($userData->getUuid());
            } catch (UserNotFound $e) {
                $company = $this->companyProvider->fetchBy($userData->getOwnerUuid());
                $user = new UserEntity($company, $userData->getUuid(), $userData->getName(), $userData->getEmail());
                $this->om->persist($user);
            }
            $user->setRoles($userData->getRoles());
            $user->clearDivisions();
            foreach ($userData->getDivisions() as $divisionUuid) {
                $division = $this->divisionProvider->fetchBy($divisionUuid);
                $user->addDivision($division);
            }
            $this->om->flush();
            $authUser = new User($user, $userData->getCurrentDivisionUuid(), $token);
            $this->dispatcher->dispatch(UserEvent::LOAD, new UserEvent($authUser));
            return $authUser;
        } catch (AccountNotFound $e) {
            throw new UnsupportedUserException();
        }
    }

    /**
     * Refreshes the user for the account interface.
     *
     * It is up to the implementation to decide if the user data should be
     * totally reloaded (e.g. from the database), or if the UserInterface
     * object can just be merged into some internal array of users / identity
     * map.
     *
     * @param UserInterface $authUser
     * @return UserInterface
     * @throws \Symfony\Component\Security\Core\Exception\UnsupportedUserException
     *
     */
    public function refreshUser(UserInterface $authUser)
    {
        if (!($authUser instanceof User)) {
            throw new UnsupportedUserException();
        }
        try {
            $user = $this->userProvider->fetchById($authUser->getEntityId());
            $authUser->setUser($user);
            $this->dispatcher->dispatch(UserEvent::LOAD, new UserEvent($authUser));
            return $authUser;
        } catch (UserNotFound $e) {
            throw new UnsupportedUserException();
        }

    }

    /**
     * Whether this provider supports the given user class.
     *
     * @param string $class
     *
     * @return bool
     */
    public function supportsClass($class)
    {
        return User::class === $class;
    }
}