<?php
namespace IssetBV\MijnIssetAuthBundle\Service\Platform;

use DateTime;
use GuzzleHttp\Client as HttpClient;
use IssetBV\MijnIssetAuthBundle\Exception\PlatformClientException;
use IssetBV\MijnIssetAuthBundle\Presenter\PaymentsPresenter;
use GuzzleHttp\Stream\Stream;
use IssetBV\MijnIssetAuthBundle\Response\GuzzleResponse;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\ServerException;

class Client
{

    /**
     * @var HttpClient
     */
    private $client;

    private $token;

    public function __construct($url, $token)
    {
        $this->token = $token;
        $this->client = new HttpClient([
            'base_url' => rtrim($url, '/') . '/'
        ]);
    }

    public function sendPayments($accountUuid, $divisionUuid, array $data)
    {
        $request = $this->client->createRequest('POST', 'api/platform/pay', array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid,
                'x-division-uuid' => $divisionUuid
            )
        ));
        
        $presenter = new PaymentsPresenter();
        $presenter->setPayments($data);
        $json = $presenter->getJson();
        
        $request->setBody(Stream::factory($json));
        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {
            
            return new GuzzleResponse($ex->getResponse());
        }
        
        return new GuzzleResponse($response);
    }

    /**
     * Send storage usage report
     * @param $divisionUuid
     * @param array $data
     * @return GuzzleResponse
     * @throws \Exception
     */
    public function sendStorageUsage($divisionUuid, array $data)
    {
        $request = $this->client->createRequest('POST', 'api/platform/storage', array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-division-uuid' => $divisionUuid
            )
        ));

        $presenter = new PaymentsPresenter();
        $presenter->setPayments($data);
        $json = $presenter->getJson();

        $request->setBody(Stream::factory($json));
        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    public function getApiKey($account, $division, $platform)
    {
        $keyData = $this->client->get('api/platform/xauth/api/key', [
            'headers' => [
                'x-my-platform' => $this->token,
                'x-account-uuid' => $account,
                'x-division-uuid' => $division,
                'x-platform' => $platform
            ]
        ])
            ->getBody()
            ->getContents();
        return json_decode($keyData)->token;
    }

    /**
     * @param $token
     * @return GuzzleResponse
     */
    public function getRemotes($accountUuid, $divisionUuid)
    {
        $request = $this->client->createRequest('GET', 'api/platform/account/remotes', array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid,
                'x-division-uuid' => $divisionUuid
            )
        ));
        
        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {
            
            return new GuzzleResponse($ex->getResponse());
        }
        
        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param $identifier
     * @return GuzzleResponse
     */
    public function getRemoteDetails($accountUuid, $divisionUuid, $identifier)
    {
        $request = $this->client->createRequest('GET', 'api/platform/account/remotes/' . $identifier, array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid,
                'x-division-uuid' => $divisionUuid
            )
        ));
        
        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {
            
            return new GuzzleResponse($ex->getResponse());
        }
        
        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param $identifier
     * @return GuzzleResponse
     */
    public function deleteRemote($accountUuid, $identifier)
    {
        $request = $this->client->createRequest('DELETE', 'api/platform/account/remotes/' . $identifier, array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid
            )
        ));
        
        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {
            
            return new GuzzleResponse($ex->getResponse());
        }
        
        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param array $remote
     * @return GuzzleResponse
     */
    public function createRemote($accountUuid, array $remote)
    {
        $request = $this->client->createRequest('POST', 'api/platform/account/remotes', array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid
            )
        ));
        $request->setBody(Stream::factory(json_encode($remote)));
        try {
            
            $response = $this->client->send($request);
        } catch (ClientException $ex) {
            
            return new GuzzleResponse($ex->getResponse());
        }
        
        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param $identifier
     * @param $remote
     * @return GuzzleResponse
     */
    public function updateRemote($accountUuid, $identifier, $remote)
    {
        $request = $this->client->createRequest('POST', 'api/platform/account/remotes/' . $identifier, array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid
            )
        ));
        
        $request->setBody(Stream::factory(json_encode($remote)));
        try {
            
            $response = $this->client->send($request);
        } catch (ClientException $ex) {
            
            return new GuzzleResponse($ex->getResponse());
        }
        
        return new GuzzleResponse($response);
    }


    /**
     * Get current balance for division
     * @param $accountUuid
     * @param $divisionUuid
     * @return GuzzleResponse
     */
    public function getBalance($accountUuid, $divisionUuid)
    {
        $request = $this->client->createRequest('GET', 'api/platform/balance', array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid,
                'x-division-uuid' => $divisionUuid
            )
        ));

        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * Get current folder size and combined storage over the past month by division
     * @param $accountUuid
     * @param $divisionUuid
     * @param DateTime $date
     * @return GuzzleResponse
     */
    public function getUsage($accountUuid, $divisionUuid, DateTime $date = null)
    {
        $url = 'api/platform/usage';
        if ($date !== null) {
            $url .= ('?date=' . $date->format('Y-m-d'));
        }

        $request = $this->client->createRequest('GET', $url, array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid,
                'x-division-uuid' => $divisionUuid
            )
        ));

        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * Ask My Isset if the current balance is sufficient to start a new task
     * @param $accountUuid
     * @param $divisionUuid
     * @return boolean
     * @throws PlatformClientException
     */
    public function hasSufficientFunds($accountUuid, $divisionUuid)
    {
        $request = $this->client->createRequest('GET', 'api/platform/has_sufficient_funds', array(
            'headers' => array(
                'x-my-platform' => $this->token,
                'x-account-uuid' => $accountUuid,
                'x-division-uuid' => $divisionUuid
            )
        ));

        try {
            $response = $this->client->send($request);

            if($response->getBody()->getSize() > 0){
                $responseMessage = $response->getBody()->read($response->getBody()->getSize());
                $responseArray = json_decode($responseMessage, true);

                if(json_last_error() !== JSON_ERROR_NONE){

                    throw new PlatformClientException("Unable to parse body received from My Isset service: " . $responseMessage);
                }

                if (isset($responseArray['has_sufficient_funds']) && is_bool($responseArray['has_sufficient_funds'])) {

                    return $responseArray['has_sufficient_funds'];
                } else {

                    throw new PlatformClientException("Unexpected body received from My Isset service: " . $responseMessage);
                }

            }else{
                throw new PlatformClientException("Empty reply from My Isset service.");
            }


        } catch (ClientException $ex) {

            throw new PlatformClientException("Negative response from My Isset service.", $ex->getResponse()->getStatusCode(), $ex);

        } catch (ConnectException $ex) {

            throw new PlatformClientException("Unable to connect to My Isset service.", 0 , $ex);
        } catch (ServerException $ex) {

            throw new PlatformClientException("Server exception received from My Isset service.", $ex->getResponse()->getStatusCode(), $ex);
        }

    }

}