<?php
namespace IssetBV\MijnIssetAuthBundle\Security\OAuth;

use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Authentication\Token\PreAuthenticatedToken;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Authentication\SimplePreAuthenticatorInterface;
use Symfony\Component\Security\Http\RememberMe\TokenBasedRememberMeServices;

/**
 * Class OAuthAuthenticator
 * @package IssetBV\MijnIssetAuthBundle\Security
 * @author <felix@isset.nl>
 */
class OAuthAuthenticator implements SimplePreAuthenticatorInterface, AuthenticationSuccessHandlerInterface
{

    protected $tokenName;

    protected $userProvider;

    protected $router;

    protected $succesRoute;

    protected $failedRoute;

    protected $cookieLifetime;

    protected $secret;

    public function __construct($tokenName, $router, $succesRoute, $failedRoute, $cookieLifetime, $secret)
    {
        $this->tokenName = $tokenName;
        $this->router = $router;
        $this->succesRoute = $succesRoute;
        $this->failedRoute = $failedRoute;
        $this->cookieLifetime = $cookieLifetime;
        $this->secret = $secret;
    }

    /**
     * @param Request $request
     * @param $providerKey
     * @return PreAuthenticatedToken
     * @throws BadCredentialsException
     */
    public function createToken(Request $request, $providerKey)
    {
        $apiKey = $request->query->get($this->tokenName);
        
        if (! $apiKey) {
            throw new BadCredentialsException('No API key found');
        }
        
        return new OAuthToken('anon.', $apiKey, $providerKey);
    }

    /**
     * @param TokenInterface $token
     * @param UserProviderInterface $userProvider
     * @param $providerKey
     * @return PreAuthenticatedToken
     * @throws AuthenticationException
     */
    public function authenticateToken(TokenInterface $token, UserProviderInterface $userProvider, $providerKey)
    {
        $this->userProvider = $userProvider;
        $apiKey = $token->getCredentials();
        $username = $userProvider->getUsernameForApiKey($apiKey);
        
        if (! $username) {
            throw new AuthenticationException(sprintf('API Key "%s" does not exist.', $apiKey));
        }
        
        $user = $userProvider->loadUserByUsername($username);
        
        $return = new OAuthToken($user, $apiKey, $providerKey, $user->getRoles());
        $return->setAuthenticated(true);
        return $return;
    }

    /**
     * @param TokenInterface $token
     * @param $providerKey
     * @return bool
     */
    public function supportsToken(TokenInterface $token, $providerKey)
    {
        return $token instanceof OAuthToken && $token->getProviderKey() === $providerKey;
    }

    /* (non-PHPdoc)
     * @see \Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface::onAuthenticationSuccess()
     */
    public function onAuthenticationSuccess(Request $request, TokenInterface $token)
    {
        $rememberMeService = new TokenBasedRememberMeServices(array(
            $this->userProvider
        ), $this->secret, $token->getProviderKey(), array(
            'path' => '/',
            'name' => 'myissetcookie',
            'domain' => null,
            'secure' => false,
            'httponly' => true,
            'lifetime' => $this->cookieLifetime, // 14 days
            'always_remember_me' => true,
            'remember_me_parameter' => '_remember_me'
        ));
        $response = new RedirectResponse($this->router->generate($this->succesRoute));
        $rememberMeService->loginSuccess($request, $response, $token);
        return $response;
    }
}