<?php
namespace IssetBV\MijnIssetAuthBundle\Security;

use \DateTime;
use Symfony\Component\Security\Core\User\UserInterface;
use IssetBV\MijnIssetAuthBundle\Service\OAuth\Client;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use IssetBV\MijnIssetAuthBundle\Security\Event\LoadUserEvent;
use Symfony\Bridge\Doctrine\RegistryInterface;
use Doctrine\Common\Persistence\ObjectManager;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use IssetBV\MijnIssetAuthBundle\Service\ClientInterface;

/**
 * Class AuthUserProvider
 * @package IssetBV\MijnIssetAuthBundle\Security
 * @author <felix@isset.nl>
 */
class AuthUserProvider implements UserProviderInterface
{

    /**
     *
     * @var RegistryInterface
     */
    private $oAuthUserRepo;

    /**
     *
     * @var ObjectManager
     */
    private $em;

    /**
     *
     * @var EventDispatcher
     */
    private $dispatcher;

    /**
     *
     * @var ClientInterface
     */
    private $client;

    /**
     *
     * @var string
     */
    private $userClass;

    /**
     * @param RegistryInterface $doctrine
     * @param EventDispatcherInterface $dispatcher
     * @param Client $client
     * @param $userClass
     * @throws \Exception
     */
    public function __construct(RegistryInterface $doctrine, EventDispatcherInterface $dispatcher, ClientInterface $client, $userClass)
    {
        if ($userClass == null || $userClass == '') {
            throw new \Exception('No user class given');
        }
        
        $this->client = $client;
        $this->oAuthUserRepo = $doctrine->getRepository($userClass);
        $this->em = $doctrine->getManager();
        $this->userClass = $userClass;
        $this->dispatcher = $dispatcher;
    }

    /**
     * @param $token
     * @return mixed
     */
    public function getUsernameForApiKey($token)
    {
        return $this->client->getAccount($token);
    }

    /**
     * @param string $data
     * @return OAuthUser
     */
    public function loadUserByUsername($data)
    {
        /** @var OAuthUser $oAuthUser */
        $oAuthUser = $this->oAuthUserRepo->findOneBy(array(
            'accountUuid' => $data->account_uuid
        ));
        
        if ($oAuthUser == null) {
            $accountUuid = $data->account_uuid;
            $ownerUuid = $data->owner_uuid;
            $username = $data->username;
            $email = $data->email;
            $roles = $data->roles;
            $divisions = $data->divisions;
            
            $userClass = $this->userClass;
            $oAuthUser = new $userClass($accountUuid, $ownerUuid, $username, $email, $roles, $divisions);
            $this->em->persist($oAuthUser);
        }
        
        if (isset($data->token) === null) {
            $oAuthUser->setToken($data->token);
        }
        
        $oAuthUser->setRoles($data->roles);
        $oAuthUser->setDivisions($data->divisions);
        $this->em->flush();
        
        $user = new AuthUser($oAuthUser, $data->division_current);
        $this->dispatcher->dispatch(LoadUserEvent::EVENT, new LoadUserEvent($user));
        return $user;
    }

    /**
     * @param UserInterface $user
     * @return UserInterface
     */
    public function refreshUser(UserInterface $user)
    {
        $oAuthUser = $this->oAuthUserRepo->find($user->getId());
        if ($oAuthUser == null) {
            throw new UnsupportedUserException();
        }
        $user->setEntity($oAuthUser);
        $this->dispatcher->dispatch(LoadUserEvent::EVENT, new LoadUserEvent($user));
        return $user;
    }

    /**
     * @param string $class
     * @return bool
     */
    public function supportsClass($class)
    {
        return $class === 'IssetBV\MijnIssetAuthBundle\Security\AuthUser';
    }
}