<?php
namespace IssetBV\MijnIssetAuthBundle\Service\OAuth;

use \GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Stream\Stream;
use IssetBV\MijnIssetAuthBundle\Presenter\PaymentsPresenter;
use IssetBV\MijnIssetAuthBundle\Service\ClientInterface;
use Exception;
use IssetBV\MijnIssetAuthBundle\Response\GuzzleResponse;

/**
 * Class Client
 * @package IssetBV\MijnIssetAuthBundle\Service\OAuth
 * @author <felix@isset.nl>
 */
class Client implements ClientInterface
{

    /**
     *
     * @var \GuzzleHttp\Client
     */
    private $client;

    /**
     * @var string
     */
    private $baseUrl;

    /**
     * @param $baseUrl
     */
    public function __construct($baseUrl)
    {
        $this->client = new HttpClient(array(
            'base_url' => $baseUrl,
            'defaults' => array(
                'timeout' => 10
            )
        ));
        $this->baseUrl = rtrim($baseUrl, '/') . '/';
    }

    /**
     * @param $token
     * @return mixed
     */
    public function getAccount($token)
    {
        $request = $this->client->get('api/oauth/account', array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));
        $data = json_decode($request->getBody()->getContents());
        $data->token = $token;
        return $data;
    }

    /**
     * @param $publicKey
     * @param $time
     * @param $hash
     * @return mixed
     */
    public function getLoginToken($publicKey, $time, $hash)
    {
        $url = sprintf('api/oauth/token/login?platform=%s&time=%s&hash=%s', urlencode($publicKey), urlencode($time), urlencode($hash));
        return json_decode($this->client->get($url)->getBody())->login_token;
    }

    /**
     * @param $token
     * @param array $data
     * @return GuzzleResponse
     */
    public function sendPayments($token, array $data)
    {
        $request = $this->client->createRequest('POST', 'api/oauth/account/payment', array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));
        
        $presenter = new PaymentsPresenter();
        $presenter->setPayments($data);
        $json = $presenter->getJson();
        
        $request->setBody(Stream::factory($json));
        try {
            
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @return GuzzleResponse
     */
    public function getRemotes($token){
        $request = $this->client->createRequest('GET', 'api/oauth/account/remotes', array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));

        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param $identifier
     * @return GuzzleResponse
     */
    public function getRemoteDetails($token, $identifier){
        $request = $this->client->createRequest('GET', 'api/oauth/account/remotes/' . $identifier, array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));

        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param $identifier
     * @return GuzzleResponse
     */
    public function deleteRemote($token, $identifier)
    {
        $request = $this->client->createRequest('DELETE', 'api/oauth/account/remotes/' . $identifier, array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));

        try {
            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param array $remote
     * @return GuzzleResponse
     */
    public function createRemote($token, array $remote){
        $request = $this->client->createRequest('POST', 'api/oauth/account/remotes', array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));
        $request->setBody(Stream::factory(json_encode($remote)));
        try {

            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }

    /**
     * @param $token
     * @param $identifier
     * @param $remote
     * @return GuzzleResponse
     */
    public function updateRemote($token, $identifier, $remote){
        $request = $this->client->createRequest('POST', 'api/oauth/account/remotes/' . $identifier, array(
            'headers' => array(
                'x-mijn-isset' => $token
            )
        ));

        $request->setBody(Stream::factory(json_encode($remote)));
        try {

            $response = $this->client->send($request);
        } catch (ClientException $ex) {

            return new GuzzleResponse($ex->getResponse());
        }

        return new GuzzleResponse($response);
    }


    /**
     * @param $loginToken
     * @param $view
     * @param $returnUrl
     * @return string
     */
    public function loginUrl($loginToken, $view, $returnUrl)
    {
        return sprintf('%soauth/login?login_token=%s&view=%s&return_url=%s', $this->baseUrl, urlencode($loginToken), urlencode($view), urlencode($returnUrl));
    }

    /**
     * @param $returnUrl
     * @return string
     */
    public function logoutUrl($returnUrl)
    {
        return sprintf('%soauth/logout?return_url=%s', $this->baseUrl, urlencode($returnUrl));
    }

}
