<?php
namespace IssetBV\MijnIssetAuthBundle\Security\OAuth;

use Symfony\Component\Security\Core\Authentication\SimplePreAuthenticatorInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Core\Exception\UsernameNotFoundException;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use IssetBV\MijnIssetAuthBundle\Security\AuthUserProvider;

/**
 * Class OAuthAuthenticator
 * @package IssetBV\MijnIssetAuthBundle\Security
 * @author <felix@isset.nl>
 */
class OAuthAuthenticator implements SimplePreAuthenticatorInterface
{

    protected $tokenName;

    /**
     * @param OAuthUserProvider $userProvider
     */
    public function __construct($tokenName)
    {
        $this->tokenName = $tokenName;
    }

    /**
     * @param Request $request
     * @param $providerKey
     * @return PreAuthenticatedToken
     * @throws BadCredentialsException
     */
    public function createToken(Request $request, $providerKey)
    {
        $apiKey = $request->query->get($this->tokenName);
        
        if (! $apiKey) {
            throw new BadCredentialsException('No API key found');
        }
        
        return new OAuthToken('anon.', $apiKey, $providerKey);
    }

    /**
     * @param TokenInterface $token
     * @param UserProviderInterface $userProvider
     * @param $providerKey
     * @return PreAuthenticatedToken
     * @throws AuthenticationException
     */
    public function authenticateToken(TokenInterface $token, UserProviderInterface $userProvider, $providerKey)
    {
        $apiKey = $token->getCredentials();
        $username = $userProvider->getUsernameForApiKey($apiKey);
        
        if (! $username) {
            throw new AuthenticationException(sprintf('API Key "%s" does not exist.', $apiKey));
        }
        
        $user = $userProvider->loadUserByUsername($username);
        
        $return = new OAuthToken($user, $apiKey, $providerKey, $user->getRoles());
        $return->setAuthenticated(true);
        return $return;
    }

    /**
     * @param TokenInterface $token
     * @param $providerKey
     * @return bool
     */
    public function supportsToken(TokenInterface $token, $providerKey)
    {
        return $token instanceof OAuthToken && $token->getProviderKey() === $providerKey;
    }

    /**
     * @param Request $request
     * @param AuthenticationException $exception
     * @return Response
     */
    public function onAuthenticationFailure(Request $request, AuthenticationException $exception)
    {
        return new Response("Authentication Failed.", 403);
    }
}