<?php

declare(strict_types=1);

namespace IssetBV\Message\MessageBundle\DependencyInjection;

use IssetBV\Message\Core\Message\Decoder\DecoderContainer;
use IssetBV\Message\Core\Message\Dequeuer\Dequeuer;
use IssetBV\Message\Core\Message\Dequeuer\DequeuerInterface;
use IssetBV\Message\Core\Message\Encoder\EncoderContainer;
use IssetBV\Message\Core\Message\Handler\HandlerContainer;
use IssetBV\Message\Core\Message\Queuer\Queuer;
use IssetBV\Message\Core\Message\Queuer\QueuerInterface;
use IssetBV\Message\Core\Queue\QueueProxy;
use IssetBV\Message\MessageBundle\Service\AMQP\AMQPConnection;
use IssetBV\Message\MessageBundle\Service\AMQP\AMQPConnectionOptions;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 */
class IssetBVMessageMessageExtension extends Extension
{
    const CONNECTION_PREFIX = 'nl.isset.message.connection.';
    const ENCODER_CONTAINER_DEFAULT = 'nl.isset.message.encoder.container.default';
    const DECODER_CONTAINER_DEFAULT = 'nl.isset.message.decoder.container.default';
    const HANDLER_CONTAINER_DEFAULT = 'nl.isset.message.handler.container.default';

    /**
     * @var ContainerBuilder
     */
    private $container;

    private $defaultQueue;

    /**
     * {@inheritdoc}
     *
     * @throws \Exception
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $this->container = $container;

        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $this->defaultQueue = $config['default_queue'];

        $this->container->register(
            self::ENCODER_CONTAINER_DEFAULT,
            EncoderContainer::class
        );

        $this->container->register(
            self::DECODER_CONTAINER_DEFAULT,
            DecoderContainer::class
        );

        $this->container->register(
            self::HANDLER_CONTAINER_DEFAULT,
            HandlerContainer::class
        );

        if (array_key_exists('connections', $config)) {
            $this->buildConnections($config['connections']);
        }

        if (array_key_exists('queues', $config)) {
            $this->buildQueues($config['queues']);
        }
    }

    private function buildQueues(array $queues)
    {
        foreach ($queues as $name => $queue) {
            $this->buildQueue($name, $queue);
        }
    }

    private function buildQueue(string $name, array $settings)
    {
        $proxyClassName = 'nl.isset.message.queue.' . $name;
        $proxyClass = $this->container->register(
            $proxyClassName,
            QueueProxy::class
        );

        $proxyClass->addArgument(new Reference(self::CONNECTION_PREFIX . $settings['connection']));
        $proxyClass->addArgument($settings['exchange']);
        $proxyClass->addArgument($settings['queue'] ?? $settings['exchange']);

        $this->buildQueuer($name, $proxyClassName);
        $this->buildDequeuer($name, $proxyClassName);
    }

    private function buildDequeuer(string $name, string $queueClassName)
    {
        $dequeuerClassName = 'nl.isset.message.dequeuer.' . $name;
        $dequeuerClass = $this->container->register(
            $dequeuerClassName,
            Dequeuer::class
        );
        $decoderContainer = $settings['decoder_container'] ?? self::DECODER_CONTAINER_DEFAULT;
        $handlerContainer = $settings['handler_container'] ?? self::HANDLER_CONTAINER_DEFAULT;

        $dequeuerClass->addArgument(new Reference($decoderContainer));
        $dequeuerClass->addArgument(new Reference($handlerContainer));
        $dequeuerClass->addArgument(new Reference($queueClassName));
        if ($this->defaultQueue === $name) {
            $this->container->setAlias(DequeuerInterface::class, $dequeuerClassName);
        }
    }

    private function buildQueuer(string $name, string $queueClassName)
    {
        $queuerClassName = 'nl.isset.message.queuer.' . $name;
        $queuerClass = $this->container->register(
            $queuerClassName,
            Queuer::class
        );
        $encoderContainer = $settings['encoder_container'] ?? self::ENCODER_CONTAINER_DEFAULT;
        $queuerClass->addArgument(new Reference($encoderContainer));
        $queuerClass->addArgument(new Reference($queueClassName));
        if ($this->defaultQueue === $name) {
            $this->container->setAlias(QueuerInterface::class, $queuerClassName);
        }
    }

    private function buildConnections(array $connections)
    {
        if (array_key_exists('isset_amqp', $connections)) {
            foreach ($connections['isset_amqp'] as $name => $settings) {
                $this->buildIssetAMQP($name, $settings);
            }
        }
    }

    private function buildIssetAMQP(string $name, array $settings)
    {
        $optionsName = 'nl.isset.message.connection.isset_ampq.options.' . $name;
        $options = $this->container->register(
            $optionsName,
            AMQPConnectionOptions::class
        );

        $options->addArgument($settings['host']);
        $options->addArgument($settings['port']);
        $options->addArgument($settings['user']);
        $options->addArgument($settings['password']);
        $options->addArgument($settings['vhost']);
        $options->addArgument($settings['insist']);
        $options->addArgument($settings['login_method']);
        $options->addArgument($settings['login_response']);
        $options->addArgument($settings['locale']);
        $options->addArgument($settings['connection_timeout']);
        $options->addArgument($settings['read_write_timeout']);
        $options->addArgument($settings['context']);
        $options->addArgument($settings['keepalive']);
        $options->addArgument($settings['heartbeat']);

        $connection = $this->container->register(
            self::CONNECTION_PREFIX . $name,
            AMQPConnection::class
        );
        $connection->addArgument(new Reference($optionsName));
    }
}
