<?php

declare(strict_types=1);

namespace IssetBV\Message\MessageBundle\Service\AMQP;

use IssetBV\Message\Core\Connection\QueueInterface;
use PhpAmqpLib\Channel\AMQPChannel;
use PhpAmqpLib\Message\AMQPMessage as PhpAMPQMessage;

class AMQPPQueue implements QueueInterface
{
    /**
     * @var AMQPChannel
     */
    private $channel;
    /**
     * @var string
     */
    private $exchange;
    /**
     * @var string
     */
    private $queue;

    private $exchangeDeclared = false;
    private $queueDeclared = false;

    public function __construct(AMQPChannel $channel, string $exchange, string $queue)
    {
        $this->channel = $channel;
        $this->exchange = $exchange;
        $this->queue = $queue;
    }

    public function send(string $message)
    {
        $this->declareExchange();
        $this->channel->basic_publish(new PhpAMPQMessage($message), $this->exchange);
    }

    public function receive()
    {
        $this->queueDeclare();
        $message = $this->channel->basic_get($this->queue);
        /* @var PhpAMPQMessage $message */
        if (null === $message) {
            return null;
        }

        return new AMQPMessage($this->channel, $message);
    }

    private function declareExchange()
    {
        if ($this->exchangeDeclared) {
            return;
        }
        $this->channel->exchange_declare($this->exchange, 'fanout', false, false, false);
        $this->exchangeDeclared = true;
    }

    private function queueDeclare()
    {
        if ($this->queueDeclared) {
            return;
        }
        $this->declareExchange();
        $this->channel->queue_declare($this->queue, false, false, false, false);
        $this->channel->queue_bind($this->queue, $this->exchange);
        $this->queueDeclared = true;
    }
}
