<?php

declare(strict_types=1);

namespace IssetBV\Message\MessageBundle\DependencyInjection;

use IssetBV\Message\Core\Connection\ConnectionFactory;
use IssetBV\Message\Core\MessageConsumerContainer;
use IssetBV\Message\Core\MessageConsumerContainerInterface;
use IssetBV\Message\Core\MessageConsumerHandler;
use IssetBV\Message\Core\MessagePublisher;
use IssetBV\Message\Core\MessagePublisherHandler;
use IssetBV\Message\Core\MessagePublisherInterface;
use IssetBV\Message\MessageBundle\Service\AMQP\AMQPConnection;
use IssetBV\Message\MessageBundle\Service\AMQP\AMQPConnectionOptions;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Loader;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 */
class IssetBVMessageMessageExtension extends Extension
{
    /**
     * @var ContainerBuilder
     */
    private $container;

    private $connectionFactoryClass;

    private $defaultConnection;

    /**
     * {@inheritdoc}
     *
     * @throws \Exception
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $this->container = $container;
        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $loader = new Loader\YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/config'));
        $loader->load('services.yml');

        $this->connectionFactoryClass = $config['connection_factory'];
        $this->defaultConnection = $config['default_connection'];

        $containerName = 'isset_bv_message_message.publisher';
        $this->container->register($containerName, MessagePublisher::class);
        $this->container->setAlias(MessagePublisher::class, $containerName);
        $this->container->setAlias(MessagePublisherInterface::class, $containerName);

        $consumerContainerName = 'isset_bv_message_message.consumer.container';
        $this->container->register($consumerContainerName, MessageConsumerContainer::class);
        $this->container->setAlias(MessageConsumerContainer::class, $consumerContainerName);
        $this->container->setAlias(MessageConsumerContainerInterface::class, $consumerContainerName);

        if (array_key_exists('connections', $config)) {
            $this->buildConnections($config['connections']);
        }

        if (array_key_exists('consumer_handlers', $config)) {
            $this->buildConsumerHandlers($config['consumer_handlers']);
        }

        if (array_key_exists('publisher_handlers', $config)) {
            $this->buildPublisherHandlers($config['publisher_handlers']);
        }
    }

    private function buildPublisherHandlers(array $publishers)
    {
        foreach ($publishers as $name => $publisher) {
            $this->buildPublisherHandler($name, $publisher);
        }
    }

    private function buildPublisherHandler(string $name, array $publisherData)
    {
        $containerName = 'isset_bv_message_message.publisher_handler.' . $name;
        $consumer = $this->container->register($containerName, MessagePublisherHandler::class);
        $consumer->addTag('isset_bv_message_message.publisher_handler');
        $consumer->addArgument(new Reference($this->connectionFactoryClass));
        $consumer->addArgument($this->defaultConnection);
        $consumer->addArgument($publisherData['exchange'] ?? $publisherData['queue']);
        $consumer->addArgument($publisherData['queue']);
        $consumer->addArgument($publisherData['handles']);
    }

    private function buildConsumerHandlers(array $consumers)
    {
        foreach ($consumers as $name => $consumer) {
            $this->buildConsumerHandler($name, $consumer);
        }
    }

    private function buildConsumerHandler(string $name, array $consumerData)
    {
        $containerName = 'isset_bv_message_message.consumer_handler.' . $name;
        $consumer = $this->container->register($containerName, MessageConsumerHandler::class);
        $consumer->addTag('isset_bv_message_message.consumer_handler');
        $consumer->addArgument($name);
        $consumer->addArgument(new Reference($this->connectionFactoryClass));
        $consumer->addArgument($this->defaultConnection);
        $consumer->addArgument($consumerData['exchange'] ?? $consumerData['queue']);
        $consumer->addArgument($consumerData['queue']);
        $consumer->addArgument($consumerData['handles']);
    }

    private function buildConnections(array $connections)
    {
        if (array_key_exists('isset_amqp', $connections)) {
            foreach ($connections['isset_amqp'] as $name => $settings) {
                $this->buildIssetAMQP($name, $settings);
            }
        }
    }

    private function buildIssetAMQP(string $name, array $settings)
    {
        $optionsName = 'isset_bv_message_message.connection.isset_ampq_options' . $name;
        $options = $this->container->register(
            $optionsName,
            AMQPConnectionOptions::class
        );
        if (null === $this->defaultConnection) {
            $this->defaultConnection = $name;
        }

        $options->addArgument($name);
        $options->addArgument($settings['host']);
        $options->addArgument($settings['port']);
        $options->addArgument($settings['user']);
        $options->addArgument($settings['password']);
        $options->addArgument($settings['vhost']);
        $options->addArgument($settings['insist']);
        $options->addArgument($settings['login_method']);
        $options->addArgument($settings['login_response']);
        $options->addArgument($settings['locale']);
        $options->addArgument($settings['connection_timeout']);
        $options->addArgument($settings['read_write_timeout']);
        $options->addArgument($settings['context']);
        $options->addArgument($settings['keepalive']);
        $options->addArgument($settings['heartbeat']);

        $connection = $this->container->register(
            'isset_bv_message_message.connection.isset_ampq_' . $name,
            AMQPConnection::class
        );
        $connection->addTag(ConnectionFactory::TAG);
        $connection->addArgument(new Reference($optionsName));
    }
}
