<?php

declare(strict_types=1);

namespace IssetBV\Message\MessageBundle\Service\AMQP;

use IssetBV\Message\Core\Connection\ConnectionInterface;
use IssetBV\Message\Core\Connection\QueueInterface;
use PhpAmqpLib\Connection\AMQPStreamConnection;

class AMQPConnection implements ConnectionInterface
{
    /**
     * @var AMQPConnectionOptions
     */
    private $options;
    /**
     * @var AMQPStreamConnection
     */
    private $connection = null;

    private $queues = [];

    public function __construct(AMQPConnectionOptions $options)
    {
        $this->options = $options;
    }

    public function getName(): string
    {
        return $this->options->getName();
    }

    public function getQueue(string $exchange, string $queue, array $options = []): QueueInterface
    {
        $name = sha1($exchange) . '##' . sha1($queue);
        if (!array_key_exists($name, $this->queues)) {
            $this->open();
            $channel = $this->connection->channel();
            $channel->exchange_declare($exchange, 'direct', false, false, false);
            $channel->queue_declare($queue, false, false, false, false);
            $channel->queue_bind($queue, $exchange);
            $this->queues[$name] = new AMQPPQueue($channel, $exchange, $queue);
        }

        return $this->queues[$name];
    }

    public function open()
    {
        if ($this->connection !== null) {
            return;
        }

        $this->connection = new AMQPStreamConnection(
            $this->options->getHost(),
            $this->options->getPort(),
            $this->options->getUser(),
            $this->options->getPassword(),
            $this->options->getVhost(),
            $this->options->isInsist(),
            $this->options->getLoginMethod(),
            $this->options->getLoginResponse(),
            $this->options->getLocale(),
            $this->options->getConnectionTimeout(),
            $this->options->getReadWriteTimeout(),
            $this->options->getContext(),
            $this->options->isKeepalive(),
            $this->options->getHeartbeat()
        );
        $this->queues = [];
    }

    public function close()
    {
        if ($this->connection !== null) {
            return;
        }
        $this->connection->close();
        $this->connection = null;
        $this->queues = [];
    }

}