<?php

declare(strict_types=1);

namespace IssetBV\Message\Core\Message\Dequeuer;

use IssetBV\Message\Core\Message\Decoder\DecoderContainerInterface;
use IssetBV\Message\Core\Message\Decoder\Exception\DecoderException;
use IssetBV\Message\Core\Message\Handler\Exception\HandlerException;
use IssetBV\Message\Core\Message\Handler\HandlerContainer;
use IssetBV\Message\Core\Message\Handler\HandlerContainerInterface;
use IssetBV\Message\Core\Message\MessageInterface;
use IssetBV\Message\Core\Queue\QueueInterface;
use IssetBV\Message\Core\Queue\QueueMessageInterface;

class Dequeuer implements DequeuerInterface
{
    /**
     * @var QueueInterface
     */
    private $queue;
    /**
     * @var DecoderContainerInterface
     */
    private $decoderContainer;
    /**
     * @var HandlerContainer
     */
    private $handlerContainer;

    public function __construct(
        DecoderContainerInterface $decoderContainer,
        HandlerContainerInterface $handlerContainer,
        QueueInterface $queue
    ) {
        $this->queue = $queue;
        $this->decoderContainer = $decoderContainer;
        $this->handlerContainer = $handlerContainer;
    }

    public function dequeue(int $wait = 0): StatsInterface
    {
        $dequeuerStats = new Stats();
        $dequeuerStats->startTimer();
        $message = $this->queue->receive($wait);
        $dequeuerStats->stopTimer();
        if (null === $message) {
            $dequeuerStats->setReceivedMessage(false);

            return $dequeuerStats;
        }
        $dequeuerStats->setReceivedMessage(true);

        $this->handleMessage($message);

        return $dequeuerStats;
    }

    private function handleMessage(QueueMessageInterface $queueMessage)
    {
        try {
            $message = $this->decode($queueMessage);

            $response = $this->handlerContainer->handle($message);
            if ($response->isSuccess()) {
                $this->queue->ack($queueMessage);
            } else {
                $this->queue->reject($queueMessage, $response->requeue());
            }
        } catch (HandlerException $e) {
            $this->queue->reject($queueMessage, false);
            throw $e;
        } catch (DecoderException $e) {
            $this->queue->reject($queueMessage, false);
            throw $e;
        }
    }

    private function decode(QueueMessageInterface $queueMessage): MessageInterface
    {
        $type = $queueMessage->getHeader(MessageInterface::HEADER_TYPE);
        $version = $queueMessage->getHeader(MessageInterface::HEADER_VERSION);
        $body = $queueMessage->getBody();

        return $this->decoderContainer->decode($type, $version, $body);
    }
}
