<?php

declare(strict_types=1);

namespace IssetBV\Message\Core;

use IssetBV\Message\Core\Connection\ConnectionFactory;
use IssetBV\Message\Core\Connection\QueueInterface;
use IssetBV\Message\Core\Exception\MessagePublisherTypeMissingException;

class MessagePublisherHandler implements MessagePublisherHandlerInterface
{
    /**
     * @var MessageEncoderInterface[]
     */
    private $publishers = [];
    /**
     * @var ConnectionFactory
     */
    private $connectionFactory;
    /**
     * @var string
     */
    private $connection;
    /**
     * @var string
     */
    private $exchange;
    /**
     * @var string
     */
    private $queue;
    /**
     * @var array
     */
    private $options;
    /**
     * @var array
     */
    private $handles;

    public function __construct(
        ConnectionFactory $connectionFactory,
        string $connection,
        string $exchange,
        string $queue,
        array $handles = [],
        array $options = []
    ) {
        $this->connectionFactory = $connectionFactory;
        $this->connection = $connection;
        $this->exchange = $exchange;
        $this->queue = $queue;
        $this->options = $options;
        $this->handles = $handles;
    }

    public function publish(MessageInterface $message)
    {
        $publisher = $this->getPublisher($message->getType());
        $messageData = $publisher->encode($message);
        $messageString = $message->getType() . ';;' . $message->getVersion() . ';;' . $messageData;
        $this->getQueue()->send($messageString);
    }

    public function add(MessageEncoderInterface $publisher)
    {
        if (0 !== count($this->handles) && in_array($publisher->getType(), $this->handles, true)) {
            return;
        }
        $this->publishers[$publisher->getType()] = $publisher;
    }

    public function handles(MessageInterface $message): bool
    {
        return array_key_exists($message->getType(), $this->publishers);
    }

    private function getQueue(): QueueInterface
    {
        $connection = $this->connectionFactory->getForName($this->connection);

        return $connection->getQueue($this->exchange, $this->queue, $this->options);
    }

    private function getPublisher(string $type): MessageEncoderInterface
    {
        if (array_key_exists($type, $this->publishers)) {
            return $this->publishers[$type];
        }

        throw new MessagePublisherTypeMissingException('Publisher missing type: ' . $type);
    }
}
