<?php

declare(strict_types=1);

namespace IssetBV\Message\Core;

use IssetBV\Message\Core\Connection\ConnectionFactory;
use IssetBV\Message\Core\Connection\QueueInterface;
use IssetBV\Message\Core\Exception\MessageConsumerTypeMissingException;
use IssetBV\Message\Core\Exception\MessageParseException;

class MessageConsumerHandler implements MessageConsumerHandlerInterface
{
    /**
     * @var MessageConsumerInterface[]
     */
    private $consumers = [];
    /**
     * @var ConnectionFactory
     */
    private $connectionFactory;
    /**
     * @var string
     */
    private $connection;
    /**
     * @var string
     */
    private $exchange;
    /**
     * @var string
     */
    private $queue;
    /**
     * @var array
     */
    private $options;
    /**
     * @var array
     */
    private $handles;
    /**
     * @var string
     */
    private $name;

    public function __construct(
        string $name,
        ConnectionFactory $connectionFactory,
        string $connection,
        string $exchange,
        string $queue,
        array $handles = [],
        array $options = []
    ) {
        $this->connectionFactory = $connectionFactory;
        $this->connection = $connection;
        $this->exchange = $exchange;
        $this->queue = $queue;
        $this->options = $options;
        $this->handles = $handles;
        $this->name = $name;
    }

    public function handleNext()
    {
        $connectionMessage = $this->getQueue()->receive();
        if (null === $connectionMessage) {
            return;
        }

        $data = explode(';;', $connectionMessage->getBody(), 3);
        if (3 !== count($data)) {
            $connectionMessage->reject();
            throw new MessageParseException('couldn\'t parse message got ' . count($data) . ' parts expected 3');
        }
        list($type, $version, $message) = $data;

        $consumer = $this->getConsumer($type);
        $decodedMessage = $consumer->decode($version, $message);
        $consumer->consume($decodedMessage);
        $connectionMessage->ack();
    }

    public function add(MessageConsumerInterface $messageConsumer)
    {
        if (0 !== count($this->handles) && in_array($messageConsumer->getType(), $this->handles, true)) {
            return;
        }
        $this->consumers[$messageConsumer->getType()] = $messageConsumer;
    }

    public function getName(): string
    {
        return $this->name;
    }

    private function getQueue(): QueueInterface
    {
        $connection = $this->connectionFactory->getForName($this->connection);

        return $connection->getQueue($this->exchange, $this->queue, $this->options);
    }

    private function getConsumer(string $type): MessageConsumerInterface
    {
        if (array_key_exists($type, $this->consumers)) {
            return $this->consumers[$type];
        }
        throw new MessageConsumerTypeMissingException('Consumer type missing in handler: ' . $type);
    }
}
